package com.amity.socialcloud.sdk.api.social.event.query

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.model.social.event.AmityEvent
import com.amity.socialcloud.sdk.model.social.event.AmityEventOriginType
import com.amity.socialcloud.sdk.model.social.event.AmityEventStatus
import com.amity.socialcloud.sdk.model.social.event.AmityEventType
import com.ekoapp.ekosdk.internal.usecase.event.GetEventPagingDataUseCase
import io.reactivex.rxjava3.core.Flowable

class AmityEventQuery internal constructor(
    private val originType: AmityEventOriginType?,
    private val originId: String?,
    private val userId: String?,
    private val status: AmityEventStatus?,
    private val type: AmityEventType?,
    private val onlyAttended: Boolean?
) {

    /**
     * Queries for events.
     * @return Flowable<PagingData<AmityEvent>>.
     */
    fun query(): Flowable<PagingData<AmityEvent>> {
        return GetEventPagingDataUseCase().execute(
            originType = originType,
            originId = originId,
            userId = userId,
            status = status,
            type = type,
            onlyAttended = onlyAttended
        )
    }

    class Builder {
        private var originType: AmityEventOriginType? = null
        private var originId: String? = null
        private var userId: String? = null
        private var status: AmityEventStatus? = null
        private var type: AmityEventType? = null
        private var onlyAttended: Boolean? = null

        /**
         * Specifies originType to filter events by origin type.
         * @return the same AmityEventQuery.Builder instance.
         */
        fun originType(originType: AmityEventOriginType): Builder {
            return apply { this.originType = originType }
        }

        /**
         * Specifies originId to filter events by origin ID.
         * @return the same AmityEventQuery.Builder instance.
         */
        fun originId(originId: String): Builder {
            return apply { this.originId = originId }
        }

        /**
         * Specifies userId to filter events created by specific user.
         * @return the same AmityEventQuery.Builder instance.
         */
        fun userId(userId: String): Builder {
            return apply { this.userId = userId }
        }

        /**
         * Specifies status to filter events by status.
         * @return the same AmityEventQuery.Builder instance.
         */
        fun status(status: AmityEventStatus): Builder {
            return apply { this.status = status }
        }

        /**
         * Specifies type to filter events by type (virtual or in_person).
         * @return the same AmityEventQuery.Builder instance.
         */
        fun type(type: AmityEventType): Builder {
            return apply { this.type = type }
        }

        /**
         * Filter to show only events the user has attended (future implementation).
         * @return the same AmityEventQuery.Builder instance.
         */
        fun onlyAttended(onlyAttended: Boolean): Builder {
            return apply { this.onlyAttended = onlyAttended }
        }

        /**
         * Instantiates AmityEventQuery with built params.
         * @return new instance of AmityEventQuery.
         */
        fun build(): AmityEventQuery {
            return AmityEventQuery(
                originType = originType,
                originId = originId,
                userId = userId,
                status = status,
                type = type,
                onlyAttended = onlyAttended
            )
        }
    }
}
