package com.amity.socialcloud.sdk.api.chat.message

import android.net.Uri
import com.amity.socialcloud.sdk.api.chat.message.create.AmityAudioMessageCreator
import com.amity.socialcloud.sdk.api.chat.message.create.AmityCustomMessageCreator
import com.amity.socialcloud.sdk.api.chat.message.create.AmityFileMessageCreator
import com.amity.socialcloud.sdk.api.chat.message.create.AmityImageMessageCreator
import com.amity.socialcloud.sdk.api.chat.message.create.AmityMessageCreationTarget
import com.amity.socialcloud.sdk.api.chat.message.create.AmityTextMessageCreator
import com.amity.socialcloud.sdk.api.chat.message.create.AmityVideoMessageCreator
import com.amity.socialcloud.sdk.api.chat.message.query.AmityMessageQuery
import com.amity.socialcloud.sdk.api.chat.message.update.AmityCustomTextMessageUpdate
import com.amity.socialcloud.sdk.api.chat.message.update.AmityTextMessageUpdate
import com.amity.socialcloud.sdk.chat.data.message.MessageRepository
import com.amity.socialcloud.sdk.model.chat.message.AmityMessage
import com.amity.socialcloud.sdk.model.chat.message.AmityMessageAttachment
import com.ekoapp.ekosdk.internal.usecase.message.FlagMessageUseCase
import com.ekoapp.ekosdk.internal.usecase.message.MessageDeleteUseCase
import com.ekoapp.ekosdk.internal.usecase.message.MessageLiveObjectUseCase
import com.ekoapp.ekosdk.internal.usecase.message.UnFlagMessageUseCase
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable


class AmityMessageRepository internal constructor() {

    /* begin_public_function
	  id: message.query
      api_style: live_collection
	*/
    fun getMessages(subChannelId: String): AmityMessageQuery.Builder {
        return AmityMessageQuery.Builder(subChannelId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: message.get
	*/
    fun getMessage(messageId: String): Flowable<AmityMessage> {
        return MessageLiveObjectUseCase().execute(messageId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: message.create.text_message
	  api_style: async
	*/
    fun createTextMessage(subChannelId: String, text: String): AmityTextMessageCreator.Builder {
        return AmityMessageCreationTarget.Builder(subChannelId)
            .with()
            .text(text)
    }
    /* end_public_function */

    /* begin_public_function
     id: message.create.image_message
     api_style: async
     */
    @Deprecated(
            message = "This function is deprecated, please use AmityChatClient.newMessageRepository().createImageMessage(subChannelId: String, attachment: AmityMessageAttachment) instead",
            replaceWith = ReplaceWith(
                    "AmityChatClient.newMessageRepository().createImageMessage(subChannelId: String, attachment: AmityMessageAttachment)",
                    "com.amity.socialcloud.sdk.api.chat.message.AmityMessageRepository"
            ),
    )
    fun createImageMessage(subChannelId: String, imageUri: Uri): AmityImageMessageCreator.Builder {
        return AmityMessageCreationTarget.Builder(subChannelId)
            .with()
            .image(AmityMessageAttachment.URL(imageUri))
    }
    /* end_public_function */
    
    fun createImageMessage(subChannelId: String, attachment: AmityMessageAttachment): AmityImageMessageCreator.Builder {
        return AmityMessageCreationTarget.Builder(subChannelId)
            .with()
            .image(attachment)
    }

    /* begin_public_function
   id: message.create.file_message
   api_style: async
   */
    @Deprecated(
            message = "This function is deprecated, please use AmityChatClient.newMessageRepository().createFileMessage(subChannelId: String, attachment: AmityMessageAttachment) instead",
            replaceWith = ReplaceWith(
                    "AmityChatClient.newMessageRepository().createFileMessage(subChannelId: String, attachment: AmityMessageAttachment)",
                    "com.amity.socialcloud.sdk.api.chat.message.AmityMessageRepository"
            ),
    )
    fun createFileMessage(subChannelId: String, fileUri: Uri): AmityFileMessageCreator.Builder {
        return AmityMessageCreationTarget.Builder(subChannelId)
            .with()
            .file(AmityMessageAttachment.URL(fileUri))
    }
    /* end_public_function */
    
    fun createFileMessage(subChannelId: String, attachment: AmityMessageAttachment): AmityFileMessageCreator.Builder {
        return AmityMessageCreationTarget.Builder(subChannelId)
            .with()
            .file(attachment)
    }

    /* begin_public_function
   id: message.create.video_message
   api_style: async
   */
    @Deprecated(
            message = "This function is deprecated, please use AmityChatClient.newMessageRepository().createVideoMessage(subChannelId: String, attachment: AmityMessageAttachment) instead",
            replaceWith = ReplaceWith(
                    "AmityChatClient.newMessageRepository().createVideoMessage(subChannelId: String, attachment: AmityMessageAttachment)",
                    "com.amity.socialcloud.sdk.api.chat.message.AmityMessageRepository"
            ),
    )
    fun createVideoMessage(subChannelId: String, videoUri: Uri): AmityVideoMessageCreator.Builder {
        return AmityMessageCreationTarget.Builder(subChannelId)
            .with()
            .video(AmityMessageAttachment.URL(videoUri))
    }
    /* end_public_function */
    
    fun createVideoMessage(subChannelId: String, attachment: AmityMessageAttachment): AmityVideoMessageCreator.Builder {
        return AmityMessageCreationTarget.Builder(subChannelId)
            .with()
            .video(attachment)
    }

    /* begin_public_function
    id: message.create.audio_message
    api_style: async
    */
    @Deprecated(
            message = "This function is deprecated, please use AmityChatClient.newMessageRepository().createAudioMessage(subChannelId: String, attachment: AmityMessageAttachment) instead",
            replaceWith = ReplaceWith(
                    "AmityChatClient.newMessageRepository().createAudioMessage(subChannelId: String, attachment: AmityMessageAttachment)",
                    "com.amity.socialcloud.sdk.api.chat.message.AmityMessageRepository"
            ),
    )
    fun createAudioMessage(subChannelId: String, audioUri: Uri): AmityAudioMessageCreator.Builder {
        return AmityMessageCreationTarget.Builder(subChannelId)
            .with()
            .audio(AmityMessageAttachment.URL(audioUri))
    }
    /* end_public_function */
    
    fun createAudioMessage(subChannelId: String, attachment: AmityMessageAttachment): AmityAudioMessageCreator.Builder {
        return AmityMessageCreationTarget.Builder(subChannelId)
                .with()
                .audio(attachment)
    }

    /* begin_public_function
    id: message.create.custom_message
    api_style: async
    */
    fun createCustomMessage(
        subChannelId: String,
        data: JsonObject
    ): AmityCustomMessageCreator.Builder {
        return AmityMessageCreationTarget.Builder(subChannelId)
            .with()
            .custom(data)
    }
    /* end_public_function */

    /* begin_public_function
        id: message.edit.text_message
        api_style: async
    */
    fun editTextMessage(messageId: String): AmityTextMessageUpdate.Builder {
        return AmityTextMessageUpdate.Builder(messageId)
    }
    /* end_public_function */

    /* begin_public_function
        id: message.edit.custom_message
        api_style: async
    */
    fun editCustomMessage(messageId: String): AmityCustomTextMessageUpdate.Builder {
        return AmityCustomTextMessageUpdate.Builder(messageId)
    }
    /* end_public_function */

    /* begin_public_function
	    id: message.soft_delete
        api_style: async
    */
    fun softDeleteMessage(messageId: String): Completable {
        return MessageDeleteUseCase().execute(messageId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: message.flag
	  api_style: async
	*/
    fun flagMessage(messageId: String): Completable {
        return FlagMessageUseCase().execute(messageId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: message.unflag
	  api_style: async
	*/
    fun unflagMessage(messageId: String): Completable {
        return UnFlagMessageUseCase().execute(messageId)
    }
    /* end_public_function */

    
    fun deleteFailedMessages(): Completable {
        return MessageRepository().deleteFailedMessages()
    }
}