package com.amity.socialcloud.sdk.social.data.category

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.social.category.query.AmityCommunityCategorySortOption
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.entity.social.category.CommunityCategoryEntity
import com.amity.socialcloud.sdk.model.social.category.AmityCommunityCategory
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.social.data.category.paging.CategoryMediator
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.entity.CommunityEntity
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable

@OptIn(ExperimentalPagingApi::class)
internal class CategoryRepository : AmityObjectRepository<CommunityCategoryEntity, AmityCommunityCategory>() {

    override fun fetchAndSave(objectId: String): Completable {
       return CategoryRemoteDataStore().getCategory(objectId)
           .flatMapCompletable {
               CategoryQueryPersister().persist(it)
           }
    }

    override fun queryFromCache(objectId: String): CommunityCategoryEntity? {
        return CategoryLocalDataStore().getCategory(objectId)
    }

    override fun mapper(): ModelMapper<CommunityCategoryEntity, AmityCommunityCategory> {
        return CategoryModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<CommunityCategoryEntity> {
        return CategoryLocalDataStore().observeCategory(objectId)
    }

    fun getCategories(categoryIds: List<String>): List<AmityCommunityCategory> {
        return CategoryLocalDataStore().getCategories(categoryIds)
            .map {
                CategoryModelMapper().map(it)
            }
    }

    fun getAllCategoriesPagingData(
        sortOption: AmityCommunityCategorySortOption,
        isDeleted: Boolean?
    ): Flowable<PagingData<AmityCommunityCategory>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = DEFAULT_PAGE_SIZE,
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = CategoryMediator(
                sortOption = sortOption,
                isDeleted = isDeleted
            ),
            pagingSourceFactory = {
                CategoryLocalDataStore().getCategoryPagingSource(
                    sortOption = sortOption,
                    isDeleted = isDeleted
                )
            },
            modelMapper = CategoryModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getLatestCategory (
        isDeleted: Boolean?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ) : Flowable<AmityCommunityCategory> {
        return CategoryLocalDataStore().getLatestCategory(
            isDeleted = isDeleted,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        )
            .map {
                CategoryModelMapper().map(it)
            }
    }

}