package com.amity.socialcloud.sdk.core

import android.os.Build
import com.amity.socialcloud.sdk.chat.data.marker.readreceipt.ReadReceiptRepository
import com.amity.socialcloud.sdk.chat.data.marker.subchannel.SubChannelMarkerRepository
import com.amity.socialcloud.sdk.chat.domain.marker.reader.MarkMessageReadUseCase
import com.amity.socialcloud.sdk.chat.domain.marker.reader.UpdateReadReceiptLatestSyncSegmentUseCase
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.NetworkConnectionEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.NetworkConnectionEvent
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.ekoapp.ekosdk.internal.ReadReceiptEntity
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.disposables.CompositeDisposable
import io.reactivex.rxjava3.schedulers.Schedulers
import java.util.LinkedList
import java.util.concurrent.TimeUnit

internal class MessageReadReceiptSyncEngine(
	sessionLifeCycleEventBus: SessionLifeCycleEventBus,
	sessionStateEventBus: SessionStateEventBus
) : SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {
	
	private var isActive: Boolean = false
	private val jobQueue: LinkedList<ReadReceiptSyncJob> = LinkedList()
	private val timer = Flowable.interval(RECEIPT_SYNC_INTERVAL, TimeUnit.SECONDS)
			.flatMapCompletable {
				syncReadReceipts()
					.onErrorComplete()
			}
			.subscribeOn(Schedulers.io())
	
	private val disposable = CompositeDisposable()
	
	init {
		ReadReceiptRepository().getUnsyncReadReceipt().forEach {
			enqueueReadReceipt(it.subChannelId, it.latestSegment)
			jobQueue.add(ReadReceiptSyncJob(it.subChannelId, it.latestSegment, ReadReceiptSyncState.CREATED, 0))
		}
		
		if (sessionStateEventBus.getCurrentEvent() == SessionState.Established) {
			startObservingReadReceiptQueue()
		}
		
		NetworkConnectionEventBus
			.observe()
			.doOnNext { event  ->
				when (event) {
					is NetworkConnectionEvent.Connected -> {
						startObservingReadReceiptQueue()
					}
					else -> {
						stopObservingReadReceiptQueue()
					}
				}
			}
			.subscribeOn(Schedulers.io())
			.subscribe()
}
	
	// Read receipt observer handling
	private fun syncReadReceipts(): Completable {
		return if (jobQueue.size == 0 || !isActive) {
			Completable.complete()
		} else {
			val readReceipt = getReadReceipt()
			if (readReceipt != null) {
				markRead(readReceipt)
			} else {
				Completable.complete()
			}
		}
	}
	
	private fun markRead(syncJob: ReadReceiptSyncJob): Completable {
		syncJob.syncState = ReadReceiptSyncState.SYNCING
		val subChannelId = syncJob.subChannelId
		val segment = syncJob.segment
		
		return MarkMessageReadUseCase().execute(subChannelId, segment)
			.doOnComplete {
				removeSynedReceipt(syncJob.subChannelId, syncJob.segment)
				UpdateReadReceiptLatestSyncSegmentUseCase().execute(syncJob.subChannelId, syncJob.segment)
			}
			.doOnError {
				if (syncJob.retryCount > MAX_RETRY) {
					jobQueue.remove(syncJob)
				} else {
					syncJob.retryCount = syncJob.retryCount + 1
					syncJob.syncState = ReadReceiptSyncState.CREATED
				}
			}
	}
	
	private fun getReadReceipt(): ReadReceiptSyncJob? {
		//  Skip when the job queue is empty
		if (jobQueue.isEmpty()) return null
		
		// Get first read receipt in the queue
		val syncJob = jobQueue.peek()
		
		// Skip when it's syncing
		if (syncJob.syncState == ReadReceiptSyncState.SYNCING) return null
		
		// Get readReceipt from cache by subChannelId
		val readReceipt = ReadReceiptRepository().getReadReceipt(syncJob.subChannelId)
		
		return if (readReceipt != null && readReceipt.latestSegment > readReceipt.latestSyncSegment) {
			syncJob.segment = readReceipt.latestSegment
			syncJob
		} else {
			// Clear all synced jobs in the job queue
			if (readReceipt != null) {
				removeSynedReceipt(readReceipt.subChannelId, readReceipt.latestSyncSegment)
				
				// Recursively getReadReceipt() until getting unsync read receipt or the job queue is empty
				getReadReceipt()
			} else {
				syncJob
			}
		}
	}
	
	private fun removeSynedReceipt(subChannelId: String, segment: Int) {
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
			jobQueue.removeIf { syncJob ->
				syncJob.subChannelId == subChannelId && syncJob.segment <= segment
			}
		} else {
			jobQueue.forEach { syncJob ->
				if (syncJob.subChannelId == subChannelId && syncJob.segment <= segment) {
					jobQueue.remove(syncJob);
				}
			}
		}
	}
	
	private fun startObservingReadReceiptQueue() {
		if (CoreClient.isUnreadCountEnable()) {
			isActive = true
			if (disposable.size() == 0 || disposable.isDisposed){
				disposable.clear()
				timer.subscribe().let(disposable::add)
			}
		}
	}
	
	private fun stopObservingReadReceiptQueue() {
		isActive = false
		if (disposable.size() > 0){
			disposable.clear()
		}
		jobQueue.forEach { receipt ->
			if (receipt.syncState == ReadReceiptSyncState.SYNCING) {
				receipt.syncState = ReadReceiptSyncState.CREATED
			}
		}
	}
	
	fun markRead(subChannelId: String, segment: Int) {
		
		// Step 1: Optimistic update of subChannelUnreadInfo.readToSegment to message.segment
		val subChannelUnreadInfo = SubChannelMarkerRepository().getSubChannelUnreadInfo(subChannelId)
		if (subChannelUnreadInfo != null && segment > subChannelUnreadInfo.readToSegment) {
			subChannelUnreadInfo.readToSegment = segment
			SubChannelMarkerRepository().saveSubChannelUnreadInfo(listOf(subChannelUnreadInfo))
		}
		
		// Step 2: Enqueue the read receipt
		enqueueReadReceipt(subChannelId, segment)
	}
	
	private fun enqueueReadReceipt(subChannelId: String, segment: Int) {
		val readReceipt: ReadReceiptEntity = ReadReceiptRepository().getReadReceipt(subChannelId) ?: ReadReceiptEntity().apply {
				this.subChannelId = subChannelId
				this.latestSegment = segment
				this.latestSyncSegment = 0
				ReadReceiptRepository().saveReadReceipts(listOf(this))
			}
		
		if (readReceipt.latestSyncSegment >= segment) {
			// Skip the job when lastSyncSegment >= segment
			return
		}
		
		var syncJob = getSyncJob(subChannelId)
		
		if (syncJob == null || syncJob.syncState == ReadReceiptSyncState.SYNCING) {
			syncJob = ReadReceiptSyncJob(subChannelId, segment, ReadReceiptSyncState.CREATED, 0)
		} else {
			if (syncJob.segment < segment) {
				syncJob.segment = segment
			}
		}
		enqueueJob(syncJob)
	}
	
	private fun getSyncJob(subChannelId: String): ReadReceiptSyncJob? {
		return jobQueue.firstOrNull { job -> job.subChannelId == subChannelId }
	}
	
	private fun enqueueJob(syncJob: ReadReceiptSyncJob) {
		if (jobQueue.size < JOB_QUEUE_SIZE) {
			jobQueue.offer(syncJob)
		} else {
			// Remove the oldest job when the queue reaches maximum capacity
			jobQueue.poll()
			jobQueue.offer(syncJob)
		}
	}
	
	override fun onSessionStateChange(sessionState: SessionState) {
		when(sessionState) {
			SessionState.Established -> {
				startObservingReadReceiptQueue()
			}
			else -> {
				stopObservingReadReceiptQueue()
			}
		}
	}
	
	override fun establish(account: EkoAccount) {
		startObservingReadReceiptQueue()
	}
	
	override fun destroy() {
		stopObservingReadReceiptQueue()
		jobQueue.clear()
	}
	
	override fun handleTokenExpire() {
		stopObservingReadReceiptQueue()
	}
	
	data class ReadReceiptSyncJob (
		val subChannelId: String,
		var segment: Int,
		var syncState: ReadReceiptSyncState,
		var retryCount: Int
	)
	
	enum class ReadReceiptSyncState {
		CREATED, SYNCING
	}
	
	companion object {
		// Max retry count for message read receipt sync
		val MAX_RETRY = 3
		
		// Max job queue size for message read receipt sync
		val JOB_QUEUE_SIZE = 120
		
		// Interval for message read receipt sync in seconds
		val RECEIPT_SYNC_INTERVAL = 1L
	}
}