package com.amity.socialcloud.sdk.api.chat.channel

import com.amity.socialcloud.sdk.api.chat.channel.create.AmityChannelCreationType
import com.amity.socialcloud.sdk.api.chat.channel.query.AmityChannelQueryType
import com.amity.socialcloud.sdk.api.chat.channel.update.AmityChannelUpdate
import com.amity.socialcloud.sdk.api.chat.member.AmityChannelModeration
import com.amity.socialcloud.sdk.api.chat.member.AmityChannelParticipation
import com.amity.socialcloud.sdk.api.chat.notification.AmityChannelNotification
import com.amity.socialcloud.sdk.chat.domain.channel.ChannelLiveObjectUseCase
import com.amity.socialcloud.sdk.chat.domain.channel.GetChannelsUseCase
import com.amity.socialcloud.sdk.chat.domain.channel.JoinChannelUseCase
import com.amity.socialcloud.sdk.chat.domain.channel.LeaveChannelUseCase
import com.amity.socialcloud.sdk.chat.domain.channel.moderation.MuteChannelUseCase
import com.amity.socialcloud.sdk.chat.domain.channel.moderation.UnMuteChannelUseCase
import com.amity.socialcloud.sdk.chat.domain.marker.channel.GetTotalChannelsUnreadInfoUseCase
import com.amity.socialcloud.sdk.chat.domain.marker.channel.GetTotalChannelsUnreadUseCase
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannel
import com.amity.socialcloud.sdk.model.core.unread.UserUnread
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import org.joda.time.Duration


class AmityChannelRepository internal constructor() {

    /* begin_public_function
	  id: channel.create
	  api_style: async
	*/
    fun createChannel(displayName: String): AmityChannelCreationType {
        return AmityChannelCreationType(displayName)
    }
    /* end_public_function */

    /* begin_public_function
	  id: channel.query
	  api_style: live_collection
	*/
    fun getChannels(): AmityChannelQueryType {
        return AmityChannelQueryType()
    }
    /* end_public_function */

    /* begin_public_function
	  id: channel.get
	*/
    fun getChannel(channelId: String): Flowable<AmityChannel> {
        return ChannelLiveObjectUseCase().execute(channelId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: channel.join
	*/
    fun joinChannel(channelId: String): Single<AmityChannel> {
        return JoinChannelUseCase().execute(channelId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: channel.leave
	*/
    fun leaveChannel(channelId: String): Completable {
        return LeaveChannelUseCase().execute(channelId)
    }
    /* end_public_function */

    @Deprecated(
        message = "This function is deprecated, please use AmityChatClient.newChannelRepository().editChannel() instead",
        replaceWith = ReplaceWith(
            "AmityChatClient.newChannelRepository().editChannel()",
            "com.amity.socialcloud.sdk.api.chat.subchannel.AmityChannelRepository"
        ),
    )
    fun updateChannel(channelId: String): AmityChannelUpdate.Builder {
        return AmityChannelUpdate.Builder(channelId)
    }

    /* begin_public_function
    id: channel.update
    api_style: async
    */
    fun editChannel(channelId: String): AmityChannelUpdate.Builder {
        return AmityChannelUpdate.Builder(channelId)
    }
    /* end_public_function */

    fun membership(channelId: String): AmityChannelParticipation {
        return AmityChannelParticipation(channelId)
    }

    fun moderation(channelId: String): AmityChannelModeration {
        return AmityChannelModeration(channelId)
    }

    @Deprecated("Use AmityCoreClient.notification.channel(channelId: String) instead")
    fun notification(channelId: String): AmityChannelNotification {
        return AmityChannelNotification(channelId)
    }

    /* begin_public_function
	  id: channel.mute
	*/
    fun muteChannel(channelId: String, timeout: Duration): Completable {
        return MuteChannelUseCase().execute(channelId, timeout)
    }
    /* end_public_function */

    /* begin_public_function
	  id: channel.unmute
	*/
    fun unmuteChannel(channelId: String): Completable {
        return UnMuteChannelUseCase().execute(channelId)
    }
    /* end_public_function */

    fun getChannels(channelIds: List<String>): Flowable<List<AmityChannel>> {
        return GetChannelsUseCase().execute(channelIds)
    }

    fun getTotalChannelsUnreadInfo(): Flowable<UserUnread> {
        return GetTotalChannelsUnreadInfoUseCase().execute()
    }

    fun getTotalChannelUnread(): Flowable<UserUnread> {
        return GetTotalChannelsUnreadUseCase().execute()
    }

}