package com.amity.socialcloud.sdk.api.social.category.query

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.model.social.category.AmityCommunityCategory
import com.amity.socialcloud.sdk.social.domain.category.query.QueryCommunityCategoryListPagingDataUseCase
import io.reactivex.rxjava3.core.Flowable

class AmityCommunityCategoryQuery private constructor(
    private val sortOption: AmityCommunityCategorySortOption,
    private val includeDeleted: Boolean?
) {

    /**
     * Queries for categories
     * @return LiveCollection of AmityCommunityCategory as Flowable<PagingData<AmityCommunityCategory>>.
     */
    fun query(): Flowable<PagingData<AmityCommunityCategory>> {
        return QueryCommunityCategoryListPagingDataUseCase().execute(sortOption, includeDeleted)
    }

    class Builder internal constructor() {
        private var isDeleted: Boolean? = null
        private var sortBy: AmityCommunityCategorySortOption = AmityCommunityCategorySortOption.NAME

        /**
         * Specifies [includeDeleted] param.
         * When [includeDeleted] is true, the result will include soft-deleted categories.
         * When [includeDeleted] is false, the result will exclude soft-deleted categories.
         * The default value of [includeDeleted] is true.
         * @return the same AmityCommunityQuery.Builder instance.
         */
        fun includeDeleted(includeDeleted: Boolean): Builder {
            return apply {
                this.isDeleted = if(includeDeleted) null else false
            }
        }

        /**
         * Specifies [sortOption] param.
         * When [sortOption] is AmityCommunityCategorySortOption.NAME, posts will be sorted by name property ASC.
         * When [sortOption] is AmityCommunityCategorySortOption.FIRST_CREATED, posts will be sorted by createdAt property ASC.
         * When [sortOption] is AmityCommunityCategorySortOption.LAST_CREATED, posts will be sorted by createdAt property DESC.
         * The default value of [sortOption] is AmityCommunityCategorySortOption.NAME.
         * @return the same AmityCommunityCategoryQuery.Builder instance.
         */
        fun sortBy(sortOption: AmityCommunityCategorySortOption): Builder {
            return apply {
                this.sortBy = sortOption
            }
        }

        /**
         * Instantiates AmityCommunityCategoryQuery with built params.
         * @return new instance of AmityCommunityCategoryQuery.
         */
        fun build(): AmityCommunityCategoryQuery {
            return AmityCommunityCategoryQuery(sortBy, isDeleted)
        }

    }

}
