package com.amity.socialcloud.sdk.model.chat.message

import android.os.Parcelable
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.api.chat.message.flag.AmityMessageFlagger
import com.amity.socialcloud.sdk.api.chat.message.reaction.AmityMessageReactionQuery
import com.amity.socialcloud.sdk.api.chat.message.update.AmityCustomTextMessageUpdate
import com.amity.socialcloud.sdk.api.chat.message.update.AmityTextMessageUpdate
import com.amity.socialcloud.sdk.api.core.events.AmityTopicSubscription
import com.amity.socialcloud.sdk.api.core.reaction.AmityReactor
import com.amity.socialcloud.sdk.chat.domain.marker.reader.MarkMessageDeliveredUseCase
import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.helper.core.mention.AmityMentionee
import com.amity.socialcloud.sdk.model.core.events.AmityTopic
import com.amity.socialcloud.sdk.model.core.file.AmityAudio
import com.amity.socialcloud.sdk.model.core.file.AmityFile
import com.amity.socialcloud.sdk.model.core.file.AmityFileInfo
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.file.AmityVideo
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionMap
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionReferenceType
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.amity.socialcloud.sdk.socket.util.EkoGson
import com.ekoapp.ekosdk.ReactorObject
import com.ekoapp.ekosdk.internal.usecase.message.GetMessageDeliveredUsersUseCase
import com.ekoapp.ekosdk.internal.usecase.message.GetMessageReadUsersUseCase
import com.ekoapp.ekosdk.internal.usecase.message.MessageDeleteUseCase
import com.google.common.base.Objects
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import kotlinx.parcelize.Parcelize
import kotlinx.parcelize.TypeParceler
import org.amity.types.ObjectId
import org.joda.time.DateTime

@Parcelize
@TypeParceler<JsonObject?, JsonObjectParceler>
data class AmityMessage internal constructor(
    private val uniqueId: String = ObjectId.get().toHexString(),
    private val messageId: String = ObjectId.get().toHexString(),
    private val subChannelId: String = "",
    private val channelId: String = "",
    private val userId: String = "",
    private val parentId: String?,
    private val channelSegment: Int = 0,
    private val childrenNumber: Int = 0,
    private val editedAt: DateTime,
    private val isDeleted: Boolean = false,
    private val flagCount: Int = 0,
    private val tags: AmityTags = AmityTags(),
    internal var myReactions: List<String> = emptyList(),
    private val reactions: AmityReactionMap = AmityReactionMap(),
    private val reactionCount: Int = 0,
    internal var user: AmityUser? = null,
    private val type: DataType = DataType.CUSTOM,
    internal val rawData: JsonObject?,
    internal var data: Data? = null,
    private val syncState: String = State.SYNCED.stateName,
    private val createdAt: DateTime,
    private val updatedAt: DateTime,
    private val metadata: JsonObject?,
    internal var mentionees: List<AmityMentionee>,
    internal var isFlaggedByMe: Boolean = false,
    internal val path: String,
    internal var readCount: Int = 0,
    internal var deliveredCount: Int = 0,
): Parcelable, ReactorObject {

    fun getMessageId(): String {
        return messageId
    }

    fun getSubChannelId(): String {
        return subChannelId
    }

    fun getChannelId(): String {
        return channelId
    }

    fun getCreatorId(): String {
        return userId
    }

    fun getParentId(): String? {
        return parentId
    }

    fun getSegment(): Int {
        return channelSegment
    }

    fun getChildCount(): Int {
        return childrenNumber
    }

    fun getEditedAt(): DateTime {
        return editedAt
    }

    fun isDeleted(): Boolean {
        return isDeleted
    }

    fun getFlagCount(): Int {
        return flagCount
    }

    fun getTags(): AmityTags {
        return tags
    }

    fun getMyReactions(): List<String> {
        return myReactions
    }

    fun getReactionMap(): AmityReactionMap {
        return reactions
    }

    fun getReactionCount(): Int {
        return reactionCount
    }

    fun getDataType(): DataType {
        return type
    }

    fun getCreatedAt(): DateTime {
        return createdAt
    }

    fun getUpdatedAt(): DateTime {
        return updatedAt
    }

    fun getCreator(): AmityUser? {
        return user
    }

    fun getData(): Data {
        return data!!
    }
    
    /* begin_public_function
	  id: message.sending_status
	*/
    fun getState(): State {
        return State.enumOf(syncState)
    }
    /* end_public_function */

    fun isEdited(): Boolean {
        return editedAt.isAfter(createdAt)
    }

    @Deprecated(
        message = "This function is deprecated, please use AmityChatClient.newMessageRepository().softDeleteMessage() instead",
        replaceWith = ReplaceWith(
            "AmityChatClient.newMessageRepository().softDeleteMessage()",
            "com.amity.socialcloud.sdk.api.chat.message.AmityMessageRepository"
        ),
    )
    fun delete(): Completable {
        return MessageDeleteUseCase().execute(messageId)
    }

    fun isFlaggedByMe(): Boolean {
        return isFlaggedByMe
    }

    fun getMetadata(): JsonObject? {
        return metadata
    }

    fun getMentionees(): List<AmityMentionee> {
        return mentionees
    }
    
    fun getReadCount(): Int {
        return readCount
    }
    
    fun getDeliveredCount(): Int {
        return deliveredCount
    }

    fun report(): AmityMessageFlagger {
        return AmityMessageFlagger(messageId)
    }

    fun react(): AmityReactor {
        return AmityReactor(AmityReactionReferenceType.MESSAGE, messageId)
    }

    fun getReactions(): AmityMessageReactionQuery.Builder {
        return AmityMessageReactionQuery.Builder(messageId)
    }

    override fun updatedAt(): DateTime {
        return updatedAt
    }

    override fun uniqueId(): String {
        return uniqueId
    }

    fun subscription(): AmityTopicSubscription {
        return AmityTopicSubscription(AmityTopic.MESSAGE(this))
    }
    
    fun markAsDelivered(): Completable {
        return MarkMessageDeliveredUseCase().execute(subChannelId, messageId)
    }
    
    fun getReadUsers(memberships: List<MessageReadMembershipFilter> = emptyList()): Flowable<PagingData<AmityUser>> {
        return  GetMessageReadUsersUseCase().execute(messageId, memberships)
    }
    
    fun getDeliveredUsers(memberships: List<MessageDeliveredMembershipFilter> = emptyList()): Flowable<PagingData<AmityUser>> {
        return  GetMessageDeliveredUsersUseCase().execute(messageId, memberships)
    }
    
    fun markRead() {
        CoreClient.markRead(subChannelId, channelSegment)
    }

    sealed class Data : Parcelable {

        @Parcelize
        class TEXT(
            private val messageId: String,
            private val text: String? = ""
        ) : Data() {

            fun getMessageId(): String {
                return messageId
            }

            fun getText(): String {
                return text ?: ""
            }


            @Deprecated(
                message = "This function is deprecated, please use AmityChatClient.newMessageRepository().editMessage() instead",
                replaceWith = ReplaceWith(
                    "AmityChatClient.newMessageRepository().editMessage()",
                    "com.amity.socialcloud.sdk.api.chat.message.AmityMessageRepository"
                ),
            )
            fun edit(): AmityTextMessageUpdate.Builder {
                return AmityTextMessageUpdate.Builder(getMessageId())
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is TEXT
                        && Objects.equal(other.messageId, messageId)
                        && Objects.equal(other.text, text))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(messageId, text)
            }
        }

        @Parcelize
        class IMAGE(
            private val messageId: String,
            private val caption: String?,
            private val file: AmityFileInfo? = null
        ) : Data() {

            fun getMessageId(): String {
                return messageId
            }

            fun getCaption(): String? {
                return caption
            }

            fun getImage(): AmityImage? {
                return file?.let {
                    it as AmityImage
                }
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is IMAGE
                        && Objects.equal(other.messageId, messageId)
                        && Objects.equal(other.caption, caption)
                        && Objects.equal(other.file, file))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(messageId, caption, file)
            }
        }

        @Parcelize
        class FILE(
            private val messageId: String,
            private val caption: String?,
            private val file: AmityFileInfo? = null
        ) : Data() {

            fun getMessageId(): String {
                return messageId
            }

            fun getCaption(): String? {
                return caption
            }

            fun getFile(): AmityFile? {
                return file?.let {
                    it as AmityFile
                }
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is FILE
                        && Objects.equal(other.messageId, messageId)
                        && Objects.equal(other.caption, caption)
                        && Objects.equal(other.file, file))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(messageId, caption, file)
            }

        }

        @Parcelize
        class AUDIO(
            private val messageId: String,
            private val file: AmityFileInfo? = null
        ) : Data() {

            fun getMessageId(): String {
                return messageId
            }

            fun getAudio(): AmityAudio? {
                return file?.let {
                    it as AmityAudio
                }
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is AUDIO
                        && Objects.equal(other.messageId, messageId)
                        && Objects.equal(other.file, file))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(messageId, file)
            }

        }
    
        @Parcelize
        class VIDEO(
            private val messageId: String,
            private val file: AmityFileInfo? = null,
            private val thumbnailImageFile: AmityImage? = null
        ) : Data() {
        
            fun getMessageId(): String {
                return messageId
            }
        
            fun getVideo(): AmityVideo? {
                return file?.let {
                    it as AmityVideo
                }
            }
            
            fun getThumbnailImage(): AmityImage? {
                return thumbnailImageFile
            }
        
            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is VIDEO
                        && Objects.equal(other.messageId, messageId)
                        && Objects.equal(other.file, file))
            }
        
            override fun hashCode(): Int {
                return Objects.hashCode(messageId, file)
            }
        
        }

        @Parcelize
        @TypeParceler<JsonObject, JsonObjectParceler>
        class CUSTOM(private val messageId: String,
                     private val data: JsonObject) : Data() {

            fun getMessageId(): String {
                return messageId
            }

            fun getRawData(): JsonObject {
                return data
            }

            fun <T> getSerializedData(clazz: Class<T>): T? {
                return EkoGson.get().fromJson(data, clazz)
            }

            @Deprecated(
                message = "This function is deprecated, please use AmityChatClient.newMessageRepository().editMessage() instead",
                replaceWith = ReplaceWith(
                    "AmityChatClient.newMessageRepository().editMessage()",
                    "com.amity.socialcloud.sdk.api.chat.message.AmityMessageRepository"
                ),
            )
            fun edit(): AmityCustomTextMessageUpdate.Builder {
                return AmityCustomTextMessageUpdate.Builder(getMessageId())
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is CUSTOM
                        && Objects.equal(other.data, data))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(data)
            }

        }

        companion object {
            internal fun from(
                messageId: String,
                dataType: DataType,
                data: JsonObject,
                file: AmityFileInfo?,
                thumbnailFile: AmityImage?
            ): Data {
                return when (dataType) {
                    DataType.TEXT -> {
                        TEXT(messageId, data.get("text")?.asString ?: "")
                    }
                    DataType.IMAGE -> {
                        val caption = data.get("caption")?.asString ?: ""
                        IMAGE(messageId, caption, file)
                    }
                    DataType.FILE -> {
                        val caption = data.get("caption")?.asString ?: ""
                        FILE(messageId, caption, file)
                    }
                    DataType.AUDIO -> {
                        AUDIO(messageId, file)
                    }
                    DataType.VIDEO -> {
                        VIDEO(messageId, file, thumbnailFile)
                    }
                    else -> {
                        CUSTOM(messageId, data)
                    }
                }
            }
        }
    }

    enum class State(val stateName: String) {
        CREATED("created"),
        UPLOADING("uploading"),
        SYNCING("syncing"),
        SYNCED("synced"),
        FAILED("failed");

        companion object {
            fun enumOf(value: String): State = values().find { it.stateName == value } ?: SYNCED
        }
    }

    enum class DataType(val apiKey: String) {
        TEXT("text"),
        IMAGE("image"),
        FILE("file"),
        AUDIO("audio"),
        VIDEO("video"),
        CUSTOM("custom");

        companion object {
            fun enumOf(value: String): DataType {
                return values().find { it.apiKey == value } ?: CUSTOM
            }
        }
    }

}