package com.amity.socialcloud.sdk.core.data.follow.paging

import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.core.data.follow.FollowQueryPersister
import com.amity.socialcloud.sdk.core.data.follow.FollowRemoteDataStore
import com.amity.socialcloud.sdk.model.core.follow.AmityFollowRelationship
import com.amity.socialcloud.sdk.model.core.follow.AmityFollowStatusFilter
import com.ekoapp.ekosdk.internal.api.dto.EkoFollowQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.data.model.EkoUserFollowStatusEntity
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import com.ekoapp.ekosdk.internal.usecase.user.FollowingReactorUseCase
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

class FollowingMediator(
    val userId: String,
    val filter: AmityFollowStatusFilter
) :
    DynamicQueryStreamMediator<EkoUserFollowStatusEntity, EkoFollowQueryDto, AmityFollowRelationship>(
        dynamicQueryStreamKeyCreator = FollowingKeyCreator(
            userId = userId,
            followStatusFilter = filter
        ),
        nonce = AmityNonce.FOLLOWING_LIST
    ) {

    override fun provideReactorPublisher(): Flowable<AmityFollowRelationship> {
        return FollowingReactorUseCase().execute(
            userId = userId,
            filter = filter
        ).toRx2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<EkoFollowQueryDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<EkoFollowQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: EkoFollowQueryDto): Completable {
        return FollowQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: EkoFollowQueryDto): QueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.paging?.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.follows.map { it.getId() }
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<EkoFollowQueryDto> {
        return if (userId == AmityCoreClient.getUserId()) {
            FollowRemoteDataStore().getMyFollowings(
                filter = filter,
                limit = limit,
                token = token
            ).toRx2()
        } else {
            FollowRemoteDataStore().getUserFollowings(
                userId = userId,
                filter = filter,
                limit = limit,
                token = token
            ).toRx2()
        }
    }
}

