package com.ekoapp.ekosdk.internal.api

import com.amity.socialcloud.sdk.AmityOkHttp.Companion.newBuilder
import com.amity.socialcloud.sdk.core.data.session.SessionApi
import com.amity.socialcloud.sdk.core.session.SessionError
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.infra.retrofit.conveter.Rx3ErrorHandlingCallAdapterFactory
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.socket.util.EkoGson
import com.ekoapp.ekosdk.internal.api.http.AmityErrorInterceptor
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import okhttp3.OkHttpClient
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import kotlin.reflect.KClass


internal class AmityHttpClient(
    sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    sessionStateEventBus: SessionStateEventBus
) :
    SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {

    init {
        observeEndpointChanges()
        companionSessionState = this.sessionState
        companionSessionStateEventBus = getSessionStateEventBus()
    }


    override fun onSessionStateChange(sessionState: SessionState) {
        companionSessionState = sessionState
        AmityLog.e("SSM3", "AmityHttpClient onSessionStateChange: $companionSessionState")
    }

    override fun establish(account: EkoAccount) {
        //nothing
        AmityLog.e("SSM3", "AmityHttpClient establish")
    }

    override fun destroy() {
        AmityLog.e("SSM3", "AmityHttpClient destroy")
        cancelAllRequests()
    }

    override fun handleTokenExpire() {
        cancelAllRequests()
    }

    private fun cancelAllRequests() {
        authOkHttpClient?.dispatcher?.cancelAll()
    }

    private fun observeEndpointChanges() {
        EkoEndpoint.observeHttpUrl()
            .doOnNext {
                retrofit = null
                authRetrofit = null
                okHttpClient = null
                authOkHttpClient = null
                apiMap.clear()
            }
            // Let it crash when go into error
            .subscribeOn(Schedulers.io())
            .subscribe()
    }

    companion object {
        private val apiMap: MutableMap<String, Any> = mutableMapOf()
        private var retrofit: Retrofit? = null
        private var authRetrofit: Retrofit? = null
        private var okHttpClient: OkHttpClient? = null
        private var authOkHttpClient: OkHttpClient? = null
        private lateinit var companionSessionStateEventBus: SessionStateEventBus
        private val apiKeyRequiredApis = setOf(
            SessionApi::class.java.name,
            EkoNotificationApi::class.java.name
        )
        private var companionSessionState: SessionState = SessionState.NotLoggedIn

        @Deprecated("Use get() instead")
        fun notification(): EkoNotificationApi {
            return get(EkoNotificationApi::class).blockingGet()
        }

        fun <T : Any> get(clazz: KClass<T>): Single<T> {
            // check if the session is established or not, if not then throw error
            if (requiresAuthentication(clazz) && companionSessionState !is SessionState.Established) {
                return Single.error(SessionError.fromState(companionSessionState))
            }
            else {
                val existingInstance = apiMap[clazz.java.name]
                return if (existingInstance != null) {
                    Single.just(existingInstance as T)
                } else {
                    val newInstance = createApi(clazz)
                    apiMap[clazz.java.name] = newInstance
                    Single.just(newInstance)
                }
            }
        }

        private fun <T : Any> createApi(clazz: KClass<T>): T {
            return if (requiresAuthentication(clazz)) {
                getAuthRetrofit().create(clazz.java)
            } else {
                getRetrofit().create(clazz.java)
            }
        }

        private fun getRetrofit(): Retrofit {
            if (retrofit == null) {
                retrofit = createRetrofit(EkoEndpoint.getHttpUrl(), getOkHttpClient())
            }
            return retrofit!!
        }

        private fun getAuthRetrofit(): Retrofit {
            if (authRetrofit == null) {
                authRetrofit = createRetrofit(EkoEndpoint.getHttpUrl(), getAuthOkHttpClient())
            }
            return authRetrofit!!
        }

        private fun getOkHttpClient(): OkHttpClient {
            if (okHttpClient == null) {
                okHttpClient = newBuilder()
                    .build()
            }
            return okHttpClient!!
        }

        private fun getAuthOkHttpClient(): OkHttpClient {
            if (authOkHttpClient == null) {
                authOkHttpClient = newBuilder().addNetworkInterceptor(
                        AmityErrorInterceptor(
                            sessionStateEventBus = companionSessionStateEventBus
                        )
                    )
                    .build()
            }
            return authOkHttpClient!!
        }

        private fun createRetrofit(url: String, httpClient: OkHttpClient): Retrofit {
            return Retrofit.Builder()
                .baseUrl(url)
                .client(httpClient)
                .addCallAdapterFactory(Rx3ErrorHandlingCallAdapterFactory.create())
                .addConverterFactory(GsonConverterFactory.create(EkoGson.get()))
                .build()
        }

        private fun <T : Any> requiresAuthentication(clazz: KClass<T>): Boolean {
            return !apiKeyRequiredApis.contains(clazz.java.name)
        }
    }

}
