package com.amity.socialcloud.sdk.social.data.category.paging

import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.api.social.category.query.AmityCommunityCategorySortOption
import com.amity.socialcloud.sdk.entity.social.category.CommunityCategoryEntity
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.model.social.category.AmityCommunityCategory
import com.amity.socialcloud.sdk.social.data.category.CategoryQueryPersister
import com.amity.socialcloud.sdk.social.data.category.CategoryRemoteDataStore
import com.amity.socialcloud.sdk.social.data.category.CategoryRepository
import com.ekoapp.ekosdk.internal.api.dto.EkoCommunityCategoryListDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

internal class CategoryMediator(
    private val sortOption: AmityCommunityCategorySortOption,
    private val isDeleted: Boolean?,
) : DynamicQueryStreamMediator<CommunityCategoryEntity, EkoCommunityCategoryListDto, AmityCommunityCategory>(

    nonce = AmityNonce.CATEGORY_LIST,
    dynamicQueryStreamKeyCreator = CategoryKeyCreator(
        sortOption = sortOption,
        isDeleted = isDeleted
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityCommunityCategory> {
        return CategoryRepository().getLatestCategory(
            isDeleted = isDeleted,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        ).toRx2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<EkoCommunityCategoryListDto> {
        return getRequest(limit = pageSize)

    }

    override fun getFetchByTokenRequest(token: String): Single<EkoCommunityCategoryListDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: EkoCommunityCategoryListDto): Completable {
        return CategoryQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: EkoCommunityCategoryListDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.token?.next,
            previous = dto.token?.previous,
            primaryKeys = dto.results?.map { it.categoryId } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<EkoCommunityCategoryListDto> {
        return CategoryRemoteDataStore().queryCategories(
            sortBy = sortOption.apiKey,
            isDeleted = isDeleted,
            options = QueryOptionsRequestParams(limit = limit, token = token)
        ).toRx2()
    }

}