package com.amity.socialcloud.sdk.social.data.storytarget

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.api.social.storytarget.AmityGlobalStoryTargetsQueryOption
import com.amity.socialcloud.sdk.entity.social.story.StoryTargetEntity
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.amity.socialcloud.sdk.model.social.story.AmityStoryTarget
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime

internal class StoryTargetLocalDataStore {

    fun saveStoryTarget(storyTargets: List<StoryTargetEntity>): Completable {
        return Completable.fromAction {
            storyTargets.forEach {
                val cache = UserDatabase.get().storyTargetDao().getStoryTarget(
                    targetType = it.targetType,
                    targetId = it.targetId
                )
                if (cache != null) {
                    it.localLastStoryExpiresAt = cache.localLastStoryExpiresAt
                    it.localLastStorySeenExpiresAt = cache.localLastStorySeenExpiresAt
                    // assign localSortingDate and hasUnseen if cache is more recent
                    val isCacheMoreRecent = if(cache.localSortingDate == null) {
                        false
                    } else if (it.localSortingDate == null){
                        true
                    } else {
                        it.localSortingDate!!.isBefore(cache.localSortingDate)
                    }

                    if(isCacheMoreRecent) {
                        it.localSortingDate = cache.localSortingDate
                        it.hasUnseen = cache.hasUnseen
                    }
                }
            }
            UserDatabase.get().storyTargetDao().save(storyTargets)
        }.subscribeOn(Schedulers.io())
    }

    fun findCacheKeys(
        option: AmityGlobalStoryTargetsQueryOption,
        expiresAfter: DateTime?
    ): List<String> {
        return UserDatabase.get().storyTargetDao().findCacheKeys(
            option = option,
            expiresAfter = expiresAfter
        )
    }

    fun createStoryTarget(
        targetType: AmityStory.TargetType,
        targetId: String,
        lastStoryExpiresAt: DateTime?,
        localLastStoryExpiresAt: DateTime?,
        lastStorySeenExpiresAt: DateTime?,
        localLastStorySeenExpiresAt: DateTime?
    ): Completable {
        return Completable.fromAction {
            val storyTarget = StoryTargetEntity().apply {
                this.uniqueId = AmityStoryTarget.generateUniqueId(targetType.apiKey, targetId)
                this.targetType = targetType.apiKey
                this.targetId = targetId
                this.lastStoryExpiresAt = lastStoryExpiresAt
                this.localLastStoryExpiresAt = localLastStoryExpiresAt
                this.lastStorySeenExpiresAt = lastStorySeenExpiresAt
                this.localLastStorySeenExpiresAt = localLastStorySeenExpiresAt
                this.localSortingDate = localLastStoryExpiresAt
                if(localLastStoryExpiresAt?.isAfter(DateTime.now()) ?: false) {
                    this.hasUnseen = true
                }
            }
            UserDatabase.get().storyTargetDao().save(listOf(storyTarget))
        }
    }

    fun getStoryTarget(targetType: AmityStory.TargetType, targetId: String): StoryTargetEntity? {
        return UserDatabase.get().storyTargetDao().getStoryTarget(
            targetType = targetType.apiKey,
            targetId = targetId
        )
    }

    fun getStoryTargetById(uniqueId: String): StoryTargetEntity? {
        return UserDatabase.get().storyTargetDao().getByIdNowImpl(uniqueId)
    }

    fun observeStoryTarget(uniqueId: String): Flowable<StoryTargetEntity> {
        return UserDatabase.get().storyTargetDao().observeStory(uniqueId)
    }

    fun updateStoryTargetLocalLastStoryExpiresAt(
        targetType: AmityStory.TargetType,
        targetId: String,
        localLastStoryExpiresAt: DateTime?
    ): Completable {
        return UserDatabase.get().storyTargetDao().updateStoryTargetLocalLastStoryExpiresAt(
            targetType = targetType.apiKey,
            targetId = targetId,
            localLastStoryExpiresAt = localLastStoryExpiresAt
        )
    }

    fun updateStoryTargetLastStoryExpiresAt(
        targetType: AmityStory.TargetType,
        targetId: String,
        lastStoryExpiresAt: DateTime?
    ): Completable {
        return UserDatabase.get().storyTargetDao().updateStoryTargetLastStoryExpiresAt(
            targetType = targetType.apiKey,
            targetId = targetId,
            lastStoryExpiresAt = lastStoryExpiresAt
        )
    }

    fun updateStoryTargetLocalLastStorySeenExpiresAt(
        targetType: AmityStory.TargetType,
        targetId: String,
        localLastStorySeenExpiresAt: DateTime?
    ): Completable {
        return UserDatabase.get().storyTargetDao().updateStoryTargetLocalLastStorySeenExpiresAt(
            targetType = targetType.apiKey,
            targetId = targetId,
            localLastStorySeenExpiresAt = localLastStorySeenExpiresAt
        )
    }

    fun updateStoryTargetLocalSortingDate(
        targetType: AmityStory.TargetType,
        targetId: String,
        hasUnseen: Boolean,
        localSortingDate: DateTime?
    ): Completable {
        return UserDatabase.get().storyTargetDao().updateStoryTargetHasUnseen(
            targetType = targetType.apiKey,
            targetId = targetId,
            hasUnseen = hasUnseen,
            localSortingDate = localSortingDate
        )
    }

    fun getStoryTargets(targets: List<Pair<AmityStory.TargetType, String>>): Flowable<List<StoryTargetEntity>> {
        return UserDatabase.get().storyTargetSinglePageDao().getStoryTargetByTargets(targets)
    }

    fun getStoryTargetPagingSource(
        option: AmityGlobalStoryTargetsQueryOption
    ): PagingSource<Int, StoryTargetEntity> {
        return UserDatabase.get().storyTargetPagingDao().getStoryTargetPagingSource(
            option = option
        )
    }

    fun getLatestStoryTargets(
        option: AmityGlobalStoryTargetsQueryOption,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<StoryTargetEntity> {
        return UserDatabase.get().storyTargetDao().getLatestStoryTargets(
            option,
            dynamicQueryStreamKeyCreator.toMap().hashCode(),
            nonce,
            DateTime.now()
        )
    }

}