package com.amity.socialcloud.sdk.chat.data.subchannel

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.ekoapp.ekosdk.internal.SubChannelEntity
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime

internal class SubChannelLocalDataStore {

    fun hasInLocal(subChannelId: String): Boolean {
        return UserDatabase.get().subChannelDao().getByIdNow(subChannelId) != null
    }

    fun saveSubChannels(subChannels: List<SubChannelEntity>): Completable {
        return Completable.fromAction {
            UserDatabase.get().subChannelDao().save(subChannels)
        }.subscribeOn(Schedulers.io())
    }

    fun getSubChannel(subChannelId: String): SubChannelEntity? {
        var subChannel: SubChannelEntity? = null
        Completable.fromCallable {
            val entity = UserDatabase.get().subChannelDao().getByIdNow(subChannelId)
            if (entity != null) {
                subChannel = entity
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return subChannel
    }

    fun observe(subChannelId: String): Flowable<SubChannelEntity> {
        return UserDatabase.get().subChannelDao().observeSubChanel(subChannelId)
            .map {
                if (it.isEmpty()) {
                    throw AmityException.create(
                        message = "item not found",
                        cause = null,
                        error = AmityError.ITEM_NOT_FOUND
                    )
                } else {
                    return@map it[0]
                }
            }
    }

    fun hardDelete(subChannelId: String): Completable {
        return UserDatabase.get().subChannelDao().deleteById(subChannelId)
    }

    fun softDelete(subChannelId: String): Completable {
        return UserDatabase.get().subChannelDao().softDeleteById(subChannelId)
    }

    fun getSubChannelPagingSource(
        channelId: String,
        isDeleted: Boolean,
    ): PagingSource<Int, SubChannelEntity> {
        return UserDatabase.get().subChannelPagingDao().getSubChannelPagingSource(
            channelId = channelId,
            isDeleted = isDeleted
        )
    }

    fun getLatestSubChannel(
        isDeleted: Boolean,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<SubChannelEntity> {
        return UserDatabase.get().subChannelDao().getLatestSubChannel(
            isDeleted,
            dynamicQueryStreamKeyCreator.toMap().hashCode(),
            nonce,
            DateTime.now()
        )
    }

    fun updateMarkerHash(subChannelId: String, hash: Int) {
        UserDatabase.get().subChannelDao().updateMarkerHash(subChannelId, hash)
    }

    fun updateUserMarkerHash(subChannelId: String, hash: Int) {
        UserDatabase.get().subChannelDao().updateUserMarkerHash(subChannelId, hash)
    }

    fun updateLastActivity(subChannelId: String): Completable {
        return Completable.fromAction {
            UserDatabase.get().subChannelDao().updateLastActivity(subChannelId, DateTime.now())
        }.subscribeOn(Schedulers.io())
    }

    fun updateLatestMessage(subChannelId: String, messageId: String): Completable {
        return Completable.fromAction {
            UserDatabase.get().subChannelDao().updateLatestMessage(subChannelId, messageId)
        }.subscribeOn(Schedulers.io())
    }

    
    fun updateMessagePreview(subChannelId: String, messagePreviewId: String): Completable {
        return Completable.fromAction {
            UserDatabase.get().subChannelDao().updateMessagePreview(subChannelId, messagePreviewId)
        }.subscribeOn(Schedulers.io())
    }
    
    fun notifyChanges(subChannelId: String) {
        UserDatabase.get().subChannelDao().notifyChanges(subChannelId)
    }
    
    fun notifySubChannelsChanges(subChannelIds: List<String>) {
        if (subChannelIds.isNotEmpty()) {
            UserDatabase.get().subChannelDao().notifySubChannelsChanges(subChannelIds, subChannelIds.hashCode())
        }
    }
}