package com.amity.socialcloud.sdk.infra.upload

import android.net.Uri
import co.amity.rxupload.extension.upload
import com.amity.socialcloud.sdk.core.data.file.RawFileModelMapper
import com.amity.socialcloud.sdk.model.core.file.AmityFileAccessType
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadInfo
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadResult
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.util.AppContext
import io.reactivex.rxjava3.android.schedulers.AndroidSchedulers
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers


private const val UPLOAD_IMAGE_PATH = "api/v4/images"
private const val MULTIPART_DATA_KEY = "files"

class AmityImageUploadService private constructor(
    private val uri: Uri,
    private val uploadId: String?,
    private val isFullImage: Boolean
) : AmityUploadService<AmityUploadResult<AmityImage>>() {

    override fun getUploadParams(): Map<String, Any> {
        return mapOf(
            AmityFileAccessType.apiKey to getAccessType().apiKey
        )
    }

    override fun getUploadHeaders(): Map<String, Any> {
        return mapOf()
    }

    override fun makeUploadServiceRequest(): Flowable<AmityUploadResult<AmityImage>> {
        return uploadContentUri(uri)
    }

    private fun uploadContentUri(uri: Uri): Flowable<AmityUploadResult<AmityImage>> {
        return uri.upload(
            context = AppContext.get(),
            path = UPLOAD_IMAGE_PATH,
            params = getUploadParams(),
            headers = getUploadHeaders(),
            id = uploadId,
            multipartDataKey = MULTIPART_DATA_KEY,
            accessType = getAccessType().apiKey,
        )
            .flatMap { fileProps ->
                if (fileProps.progress == 100) {
                    Single.fromCallable {
                        val fileEntity = parseEkoFileEntity(fileProps.responseBody)
                        fileEntity.filePath = fileProps.uri.path
                        val fileDao = UserDatabase.get().fileDao()
                        fileDao.insert(fileEntity)
                        RawFileModelMapper().map(fileEntity)
                    }
                        .subscribeOn(Schedulers.io())
                        .observeOn(AndroidSchedulers.mainThread())
                        .flatMapPublisher {
                            Flowable.just(
                                AmityUploadResult.COMPLETE(
                                    AmityImage(it)
                                )
                            )
                        }
                } else {
                    Flowable.just(AmityUploadResult.PROGRESS(AmityUploadInfo(fileProps)))
                }
            }
            .onErrorReturn {
                val exception = parseErrorResponse(it)
                AmityUploadResult.ERROR(exception)
            }
    }

    class Builder {

        private lateinit var uri: Uri
        private var uploadId: String? = null
        private var isFullImage = false

        internal fun fileUri(uri: Uri): Builder {
            this.uri = uri
            return this
        }

        fun uploadId(uploadId: String): Builder {
            this.uploadId = uploadId
            return this
        }

        fun isFullImage(isFullImage: Boolean): Builder {
            this.isFullImage = isFullImage
            return this
        }

        fun build(): AmityImageUploadService {
            return AmityImageUploadService(uri, uploadId, isFullImage)
        }
    }

}