package com.amity.socialcloud.sdk.core.data.user

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipLocalDataStore
import com.amity.socialcloud.sdk.chat.data.message.MessageLocalDataStore
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.data.user.paging.UserMediator
import com.amity.socialcloud.sdk.entity.core.user.UserEntity
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.amity.socialcloud.sdk.api.core.user.search.AmityUserSortOption
import com.amity.socialcloud.sdk.model.core.user.UserUpdateOption
import com.amity.socialcloud.sdk.social.data.community.membership.CommunityMembershipLocalDataStore
import com.ekoapp.ekosdk.internal.api.dto.EkoUserListDto
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.core.data.follow.FollowLocalDataStore
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

@OptIn(ExperimentalPagingApi::class)
internal class UserRepository : AmityObjectRepository<UserEntity, AmityUser>() {

    override fun fetchAndSave(objectId: String): Completable {
        return UserRemoteDataStore().getUser(objectId)
            .flatMap {
                UserQueryPersister().persist(it)
                    .andThen(Single.just((it.users)))
            }
            .doOnSuccess {
                if (it.isNullOrEmpty()) {
                    throw AmityException.create(
                        message = AmityError.ITEM_NOT_FOUND.name,
                        cause = null,
                        error = AmityError.ITEM_NOT_FOUND
                    )
                }
            }
            .ignoreElement()
    }

    override fun queryFromCache(objectId: String): UserEntity? {
        return UserLocalDataStore().getUser(objectId)
    }

    override fun mapper(): ModelMapper<UserEntity, AmityUser> {
        return UserModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<UserEntity> {
        return UserLocalDataStore().observeUser(objectId)
    }

    fun getUserPagingData(
        keyword: String?,
        sortBy: AmityUserSortOption
    ): Flowable<PagingData<AmityUser>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = UserMediator(
                keyword = keyword,
                sortBy = sortBy
            ),
            pagingSourceFactory = {
                UserLocalDataStore().getUserPagingSource(
                    keyword = keyword,
                    sortBy = sortBy
                )
            },
            modelMapper = UserModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getUser(userId: String): AmityUser? {
        return UserLocalDataStore().getUser(userId)?.let {
            UserModelMapper().map(it)
        }
    }
    
    fun getUserNow(userId: String): AmityUser? {
        return UserLocalDataStore().getUserNow(userId)?.let {
            UserModelMapper().map(it)
        }
    }

    fun getUserByIds(userIds: List<String>): Flowable<List<AmityUser>> {
        return UserRemoteDataStore().getUserByIds(userIds)
            .flatMap {
                UserQueryPersister().persist(it)
                    .andThen(Single.just((it.users.map { it.userId })))
            }
            .flatMapPublisher { ids ->
                UserLocalDataStore().getUserByIds(ids)
                    .map { users ->
                        users.map {
                            UserModelMapper().map(it)
                        }
                    }
            }
    }

    fun updateUser(userId: String, option: UserUpdateOption): Single<AmityUser> {
        return UserRemoteDataStore().updateUser(userId, option)
            .flatMap {
                UserQueryPersister().persist(it)
                    .andThen(Single.just((it.users.first().userId)))
            }.flatMap { userId ->
                getUser(userId)?.let { Single.just(it) } ?: Single.never()
            }
    }

    fun hasInLocal(userId: String): Boolean {
        return UserLocalDataStore().getUser(userId) != null
    }

    fun flagUser(userId: String): Completable {
        return UserRemoteDataStore().flagUser(userId)
            .flatMapCompletable {
                UserQueryPersister().persist(it)
                    .andThen(notifyUserUpdate(userId))
            }
    }

    fun unflagUser(userId: String): Completable {
        return UserRemoteDataStore().unflagUser(userId)
            .flatMapCompletable {
                UserQueryPersister().persist(it)
                    .andThen(notifyUserUpdate(userId))
            }
    }

    fun isFlaggedByMe(userId: String): Single<Boolean> {
        return UserRemoteDataStore().isFlaggedByMe(userId)
            .map { it.get("isFlagByMe").asBoolean }
    }
    
    fun getUserDtoByIds(userIds: List<String>): Single<EkoUserListDto> {
        return UserRemoteDataStore().getUserByIds(userIds)
    }

    //  dummy update, for triggering user update tied to related entity
    internal fun notifyUserUpdate(userId: String): Completable {
        return Completable.fromAction {
            MessageLocalDataStore().notifyUserUpdate(userId)
            ChannelMembershipLocalDataStore().notifyUserUpdate(userId)
            CommunityMembershipLocalDataStore().notifyUserUpdate(userId)
            FollowLocalDataStore().notifyUserUpdate(userId)
        }
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }
}