package com.amity.socialcloud.sdk.api.social.post

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.api.core.ExperimentalAmityApi
import com.amity.socialcloud.sdk.api.social.AmitySocialClient
import com.amity.socialcloud.sdk.api.social.comment.query.AmityCommentQuery
import com.amity.socialcloud.sdk.api.social.post.create.datatype.*
import com.amity.socialcloud.sdk.api.social.post.create.target.AmityPostCreationTarget
import com.amity.socialcloud.sdk.api.social.post.query.AmityPostQueryTarget
import com.amity.socialcloud.sdk.api.social.post.reaction.AmityPostReactionQuery
import com.amity.socialcloud.sdk.api.social.post.review.AmityPostReviewer
import com.amity.socialcloud.sdk.api.social.post.update.AmityPostUpdate
import com.amity.socialcloud.sdk.model.core.file.AmityFile
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.file.AmityVideo
import com.amity.socialcloud.sdk.model.core.pin.AmityPinSortByOptions
import com.amity.socialcloud.sdk.model.core.pin.AmityPinnedPost
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.social.domain.post.create.*
import com.amity.socialcloud.sdk.social.domain.post.flag.IsPostFlaggedByMeUseCase
import com.amity.socialcloud.sdk.social.domain.post.getbyids.PostGetByIdsUseCase
import com.amity.socialcloud.sdk.social.domain.post.update.PostEditUseCase
import com.ekoapp.ekosdk.internal.usecase.post.*
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

class AmityPostRepository internal constructor() {

    /* begin_public_function
	  id: post.get
	*/
    /**
     * Gets a LiveObject of post
     * @param [postId] a postId
     * @return LiveObject of post as Flowable<AmityPost>
     */
    fun getPost(postId: String): Flowable<AmityPost> {
        return PostLiveObjectUseCase().execute(postId)
    }
    /* end_public_function */

    /* begin_public_function
      id: post.get_by_ids
	  api_style: live_collection
   */
    /**
     * Gets a LiveCollection of posts
     * @param [postIds] an array of postId
     * @return LiveCollection of post as Flowable<List<AmityPost>>
     */
    fun getPostByIds(postIds: Set<String>): Flowable<List<AmityPost>> {
        return PostGetByIdsUseCase().execute(postIds)
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.query
	  api_style: live_collection
	*/
    /**
     * Initiates a function chain to query for post.
     * @return new instance of AmityPostQueryTarget.
     */
    fun getPosts(): AmityPostQueryTarget {
        return AmityPostQueryTarget()
    }
    /* end_public_function */


    /**
     * Initiates a function chain to create a post.
     * @param [postId] a postId
     * @return new instance of AmityPostCreationTarget.
     */
    @Deprecated("Use dedicated post creation function with dedicated post type instead")
    fun createPost(): AmityPostCreationTarget {
        return AmityPostCreationTarget()
    }

    /* begin_public_function
	  id: post.create.text_post
	  api_style: async
	*/
    /**
     * create a text post
     * @return Async post as Single<AmityPost>
     */
    fun createTextPost(
        target: AmityPost.Target,
        text: String,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null
    ): Single<AmityPost> {
        return TextPostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            text = text,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList()
        )
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.create.image_post
	  api_style: async
	*/
    /**
     * create an image post
     * @return Async post as Single<AmityPost>
     */
    fun createImagePost(
        target: AmityPost.Target,
        images: Set<AmityImage>,
        text: String = "",
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null
    ): Single<AmityPost> {
        return ImagePostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            images = images.toList(),
            text = text,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList()
        )
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.create.file_post
	  api_style: async
	*/
    /**
     * create a file post
     * @return Async post as Single<AmityPost>
     */
    fun createFilePost(
        target: AmityPost.Target,
        files: Set<AmityFile>,
        text: String = "",
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null
    ): Single<AmityPost> {
        return FilePostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            files = files.toList(),
            text = text,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList()
        )
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.create.video_post
	  api_style: async
	*/
    /**
     * create a video post
     * @return Async post as Single<AmityPost>
     */
    fun createVideoPost(
        target: AmityPost.Target,
        videos: Set<AmityVideo>,
        text: String = "",
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null
    ): Single<AmityPost> {
        return VideoPostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            videos = videos.toList(),
            text = text,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList()
        )
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.create.poll_post
	  api_style: async
	*/
    /**
     * create a poll post
     * @return Async post as Single<AmityPost>
     */
    fun createPollPost(
        target: AmityPost.Target,
        pollId: String,
        text: String = "",
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null
    ): Single<AmityPost> {
        return PollPostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            pollId = pollId,
            text = text,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList()
        )
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.create.livestream_post
	  api_style: async
	*/
    /**
     * create a live stream post
     * @return Async post as Single<AmityPost>
     */
    fun createLiveStreamPost(
        target: AmityPost.Target,
        streamId: String,
        text: String = "",
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null
    ): Single<AmityPost> {
        return LiveStreamPostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            streamId = streamId,
            text = text,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList()
        )
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.create.custom_post
	  api_style: async
	*/
    /**
     * create a custom post
     * @return Async post as Single<AmityPost>
     */
    fun createCustomPost(
        target: AmityPost.Target,
        customDataType: String,
        data: JsonObject,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null
    ): Single<AmityPost> {
        return CustomPostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            dataType = AmityPost.DataType.CUSTOM(customDataType),
            data = data,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList()
        )
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.edit
	  api_style: async
	*/
    /**
     * Initiates a function chain to edit a post.
     * @param [postId] a postId
     * @return new instance of AmityPostUpdate.Builder.
     */
    fun editPost(postId: String): AmityPostUpdate.Builder {
        return AmityPostUpdate.Builder(postId)
    }
    /* end_public_function */

    /**
     * Initiates a function chain to update a post.
     * @param [postId] a postId
     * @return new instance of AmityPostUpdate.Builder.
     */
    @Deprecated("Use editPost(postId: String) instead")
    fun updatePost(postId: String): AmityPostUpdate.Builder {
        return editPost(postId)
    }

    /* begin_public_function
     id: post.edit.custom_post
     api_style: async
   */
    /**
     * edit a custom post
     * @param [postId] a postId
     * @return Async as Completable
     */
    fun editCustomPost(
        postId: String,
        data: JsonObject,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null
    ): Completable {
        return PostEditUseCase().execute(
            postId = postId,
            data = data,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList()
        )
    }
    /* end_public_function */

    /**
     * update a custom post
     * @param [postId] a postId
     * @return Async as Completable
     */
    @Deprecated("Use editCustomPost(...) instead")
    fun updateCustomPost(
        postId: String,
        data: JsonObject,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null
    ): Completable {
        return editCustomPost(
            postId = postId,
            data = data,
            metadata = metadata,
            mentionUserIds = mentionUserIds
        )
    }

    /* begin_public_function
	  id: post.soft_delete
	*/
    /**
     * soft-deletes a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun softDeletePost(postId: String): Completable {
        return PostDeleteUseCase().execute(postId, false)
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.hard_delete
	*/
    /**
     * hard-deletes a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun hardDeletePost(postId: String): Completable {
        return PostDeleteUseCase().execute(postId, true)
    }
    /* end_public_function */

    @Deprecated("Use approvePost() or declinePost() instead")
    fun reviewPost(postId: String): AmityPostReviewer {
        return AmityPostReviewer(postId)
    }

    fun getReactions(postId: String): AmityPostReactionQuery.Builder {
        return AmityPostReactionQuery.Builder(postId)
    }

    fun getComments(postId: String): AmityCommentQuery.Builder {
        return AmitySocialClient.newCommentRepository().getComments().post(postId)
    }

    /* begin_public_function
	  id: post.approve
	*/
    /**
     * approves a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun approvePost(postId: String): Completable {
        return PostApproveUseCase().execute(postId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.decline
	*/
    /**
     * declines a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun declinePost(postId: String): Completable {
        return PostDeclineUseCase().execute(postId)
    }
    /* end_public_function */

    /* begin_public_function
      id: post.flag
    */
    /**
     * flags a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun flagPost(postId: String): Completable {
        return FlagPostUseCase().execute(postId)
    }
    /* end_public_function */

    /* begin_public_function
      id: post.unflag
    */
    /**
     * unflags a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun unflagPost(postId: String): Completable {
        return UnFlagPostUseCase().execute(postId)
    }
    /* end_public_function */

    /* begin_public_function
      id: post.check_flag_by_me
    */
    @ExperimentalAmityApi
    fun isPostFlaggedByMe(postId: String): Single<Boolean> {
        return Single.fromCallable { IsPostFlaggedByMeUseCase().execute(postId) }
    }
    /* end_public_function */


    fun getPinnedPosts(
        communityId: String,
        placement: String? = null,
        sortByOptions: AmityPinSortByOptions = AmityPinSortByOptions.lastPinned
    ): Flowable<PagingData<AmityPinnedPost>> {
        return PinnedPostQueryUseCase().execute(communityId, placement, sortByOptions)
    }

    private fun semanticSearchPosts(
        query: String,
        targetType: AmityPost.TargetType?,
        targetId: String?,
        postTypes: List<AmityPost.DataType> = listOf(AmityPost.DataType.TEXT, AmityPost.DataType.IMAGE),
        matchingOnlyParentPosts: Boolean = true,
    ): Flowable<PagingData<AmityPost>> {
        return SemanticSearchPostUseCase().execute(
            query = query,
            targetType = targetType,
            targetId = targetId,
            matchingOnlyParentPost = matchingOnlyParentPosts,
            postTypes = postTypes
        )
    }

}
