package com.amity.socialcloud.sdk.social.data.community.membership.paging

import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.api.social.member.query.AmityCommunityMembershipSortOption
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.model.core.role.AmityRoles
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMember
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMembership
import com.amity.socialcloud.sdk.social.data.community.CommunityQueryPersister
import com.amity.socialcloud.sdk.social.data.community.membership.CommunityMembershipRemoteDataStore
import com.ekoapp.ekosdk.internal.api.dto.CommunityListQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.entity.CommunityMembershipEntity
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

internal class CommunityMemberSearchMediator(
    private val communityId: String,
    private val roles: AmityRoles,
    private val filters: List<AmityCommunityMembership>?,
    private val sortBy: AmityCommunityMembershipSortOption,
    private val keyword: String?
) : DynamicQueryStreamMediator<CommunityMembershipEntity, CommunityListQueryDto, AmityCommunityMember>(
    nonce = AmityNonce.COMMUNITY_MEMBERSHIP_SEARCH,
    dynamicQueryStreamKeyCreator = CommunityMemberSearchKeyCreator(
        communityId = communityId,
        roles = roles,
        filters = filters,
        sortBy = sortBy,
        keyword = keyword
    ),
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityCommunityMember> {
        return Flowable.empty()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<CommunityListQueryDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<CommunityListQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: CommunityListQueryDto): Completable {
        return CommunityQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: CommunityListQueryDto): QueryStreamQueryToken {
        return QueryStreamQueryToken(
            queryStreamKeyCreator = CommunityMemberSearchKeyCreator(
                communityId = communityId,
                roles = roles,
                filters = filters,
                sortBy = sortBy,
                keyword = keyword
            ),
            next = dto.paging?.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.communityUsers.map { user -> user.userId ?: "" }
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<CommunityListQueryDto> {
        return CommunityMembershipRemoteDataStore().searchCommunityUsers(
            communityId = communityId,
            keyword = keyword,
            roles = roles.takeIf { it.isNotEmpty() },
            memberships = filters?.map { it.value },
            sortBy = sortBy.apiKey,
            options = QueryOptionsRequestParams(limit, token)
        ).toRx2()
    }

}