package com.amity.socialcloud.sdk.social.data.community

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.social.community.query.AmityCommunitySortOption
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityFilter
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityStorySettings
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.social.data.community.paging.CommunityQueryMediator
import com.amity.socialcloud.sdk.social.data.community.paging.CommunitySearchMediator
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.entity.CommunityEntity
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagerCreator
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

@OptIn(ExperimentalPagingApi::class)
internal class CommunityRepository : AmityObjectRepository<CommunityEntity, AmityCommunity>() {

    override fun fetchAndSave(objectId: String): Completable {
        return CommunityRemoteDataStore().getCommunity(objectId)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    override fun queryFromCache(objectId: String): CommunityEntity? {
        return CommunityLocalDataStore().getCommunityById(objectId)
    }

    override fun mapper(): ModelMapper<CommunityEntity, AmityCommunity> {
        return CommunityModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<CommunityEntity> {
        return CommunityLocalDataStore().observeCommunity(objectId)
    }

    fun getCommunityPagingData(
        keyword: String,
        categoryId: String,
        filter: AmityCommunityFilter,
        sortBy: AmityCommunitySortOption,
        isDeleted: Boolean?
    ): Flowable<PagingData<AmityCommunity>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = EkoObjectRepository.DEFAULT_PAGE_SIZE,
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = CommunityQueryMediator(
                keyword = keyword,
                categoryId = categoryId,
                filter = filter,
                sortOption = sortBy,
                isDeleted = isDeleted
            ),
            pagingSourceFactory = {
                CommunityLocalDataStore().getCommunityPagingSource(
                    keyword = keyword,
                    categoryId = categoryId,
                    filter = filter,
                    sortOption = sortBy,
                    isDeleted = isDeleted,
                )
            },
            modelMapper = CommunityModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun searchCommunityPagingData(
        keyword: String,
        categoryId: String,
        filter: AmityCommunityFilter,
        sortBy: AmityCommunitySortOption,
        isDeleted: Boolean?
    ): Flowable<PagingData<AmityCommunity>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = DEFAULT_PAGE_SIZE,
                enablePlaceholders = true
            ),
            queryStreamMediator = CommunitySearchMediator(
                keyword = keyword,
                categoryId = categoryId,
                filter = filter,
                sortOption = sortBy,
                isDeleted = isDeleted
            ),
            pagingSourceFactory = {
                CommunityLocalDataStore().searchCommunityPagingSource(
                    keyword = keyword,
                    categoryId = categoryId,
                    filter = filter,
                    sortOption = sortBy,
                    isDeleted = isDeleted,
                )
            },
            modelMapper = CommunityModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getRecommendedCommunities(): Flowable<List<AmityCommunity>> {
        return CommunityRemoteDataStore().getRecommendedCommunities(QueryOptionsRequestParams(limit = DEFAULT_PAGE_SIZE))
            .flatMap {
                CommunityQueryPersister().persist(it)
                    .andThen(Single.just(it).map { it.communities.map { it.communityId!! } })
            }
            .flatMapPublisher {
                CommunityLocalDataStore().getCommunities(it)
                    .map { entities ->
                        entities.map {
                            CommunityModelMapper().map(it)
                        }
                    }
            }
    }

    fun getTrendingCommunities(): Flowable<List<AmityCommunity>> {
        return CommunityRemoteDataStore().getTrendingCommunities(QueryOptionsRequestParams(limit = DEFAULT_PAGE_SIZE))
            .flatMap {
                CommunityQueryPersister().persist(it)
                    .andThen(Single.just(it).map { it.communities.map { it.communityId!! } })
            }
            .flatMapPublisher {
                CommunityLocalDataStore().getCommunities(it)
                    .map { entities ->
                        entities.map {
                            CommunityModelMapper().map(it)
                        }
                    }
            }
    }

    fun createCommunity(
        displayName: String,
        description: String?,
        categoryIds: List<String>?,
        isPublic: Boolean?,
        metadata: JsonObject?,
        userIds: List<String>?,
        avatarFileId: String?,
        needApprovalOnPostCreation: Boolean?,
        onlyAdminCanPost: Boolean?,
        storySettings: AmityCommunityStorySettings?
    ): Single<AmityCommunity> {
        return CommunityRemoteDataStore().createCommunity(
            displayName = displayName,
            description = description,
            categoryIds = categoryIds,
            isPublic = isPublic,
            metadata = metadata,
            userIds = userIds,
            avatarFileId = avatarFileId,
            needApprovalOnPostCreation = needApprovalOnPostCreation,
            onlyAdminCanPost = onlyAdminCanPost,
            storySettings = storySettings
        ).flatMap { dto ->
            CommunityQueryPersister().persist(dto)
                .andThen(
                    dto.communities.firstOrNull()?.communityId?.let { communityId ->
                        CommunityLocalDataStore().observeCommunity(communityId).firstOrError()
                            .map { CommunityModelMapper().map(it) }
                    } ?: Single.error(
                        AmityException.create(
                            "corrupted payload",
                            null,
                            AmityError.UNKNOWN
                        )
                    )
                )
        }
    }


    fun updateCommunity(
        communityId: String,
        displayName: String?,
        description: String?,
        categoryIds: List<String>?,
        isPublic: Boolean?,
        metadata: JsonObject?,
        avatarFileId: String?,
        needApprovalOnPostCreation: Boolean?,
        onlyAdminCanPost: Boolean?,
        storySettings: AmityCommunityStorySettings?
    ): Single<AmityCommunity> {
        return CommunityRemoteDataStore().updateCommunity(
            communityId = communityId,
            displayName = displayName,
            description = description,
            categoryIds = categoryIds,
            isPublic = isPublic,
            metadata = metadata,
            avatarFileId = avatarFileId,
            needApprovalOnPostCreation = needApprovalOnPostCreation,
            onlyAdminCanPost = onlyAdminCanPost,
            storySettings = storySettings
        ).flatMap { dto ->
            CommunityQueryPersister().persist(dto)
                .andThen(dto.communities.firstOrNull()?.communityId?.let { communityId ->
                    CommunityLocalDataStore().observeCommunity(communityId)
                        .firstOrError()
                        .map { CommunityModelMapper().map(it) }
                } ?: Single.error(
                    AmityException.create(
                        "corrupted payload",
                        null,
                        AmityError.UNKNOWN
                    )
                ))
        }
    }

    // To rename to "getCommunity"
    fun getCommunityById(communityId: String): AmityCommunity? {
        val community = CommunityLocalDataStore().getCommunityById(communityId)
        return if (community == null) null else CommunityModelMapper().map(community)
    }


    fun joinCommunity(communityId: String): Completable {
        return CommunityRemoteDataStore().joinCommunity(communityId)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun leaveCommunity(communityId: String): Completable {
        return CommunityRemoteDataStore().leaveCommunity(communityId)
            .flatMapCompletable { CommunityQueryPersister().persist(it) }
    }

    fun deleteCommunity(communityId: String): Completable {
        return CommunityRemoteDataStore().deleteCommunity(communityId)
            .flatMapCompletable {
                when (it.isSuccess) {
                    true -> CommunityLocalDataStore().deleteCommunity(communityId)
                    else -> Completable.complete()
                }
            }
    }

    fun getLatestCommunity(
        categoryId: String,
        filter: AmityCommunityFilter,
        isDeleted: Boolean?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<AmityCommunity> {
        return CommunityLocalDataStore().getLatestCommunity(
            categoryId,
            filter,
            isDeleted,
            dynamicQueryStreamKeyCreator,
            nonce
        ).map {
            CommunityModelMapper().map(it)
        }
    }

    fun getPostCount(targetId: String, feedType: AmityFeedType): Flowable<Int> {
        return CommunityLocalDataStore().getPostCount(targetId, feedType)
    }

}