package com.amity.socialcloud.sdk.infra.upload

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Matrix
import android.media.ExifInterface
import android.net.Uri
import android.os.Build
import co.amity.rxupload.extension.upload
import com.amity.socialcloud.sdk.core.data.file.RawFileModelMapper
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadInfo
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadResult
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.util.AppContext
import io.reactivex.rxjava3.android.schedulers.AndroidSchedulers
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import java.io.File
import java.io.FileOutputStream
import java.io.InputStream
import java.util.UUID

private const val UPLOAD_IMAGE_PATH = "api/v4/images"
private const val MULTIPART_DATA_KEY = "files"

class AmityImageUploadService private constructor(
    private val uri: Uri,
    private val uploadId: String?,
    private val isFullImage: Boolean
) : AmityUploadService<AmityUploadResult<AmityImage>>() {

    override fun getUploadParams(): Map<String, Any> {
        return mapOf()
    }

    override fun getUploadHeaders(): Map<String, Any> {
        return mapOf()
    }

    override fun makeUploadServiceRequest(): Flowable<AmityUploadResult<AmityImage>> {
        return uploadContentUri(uri)
    }

    private fun uploadContentUri(uri: Uri): Flowable<AmityUploadResult<AmityImage>> {
        return rewriteImageFile(AppContext.get(), uri)
            .flatMapPublisher { file ->
                val bitmapUri = Uri.fromFile(file)
                bitmapUri.upload(
                    context = AppContext.get(),
                    path = UPLOAD_IMAGE_PATH,
                    params = getUploadParams(),
                    headers = getUploadHeaders(),
                    id = uploadId,
                    multipartDataKey = MULTIPART_DATA_KEY
                )
                    .flatMap { fileProps ->
                        if (fileProps.progress == 100) {
                            deleteFile(file)
                            Single.fromCallable {
                                val fileEntity = parseEkoFileEntity(fileProps.responseBody)
                                fileEntity.filePath = fileProps.uri.path
                                val fileDao = UserDatabase.get().fileDao()
                                fileDao.insert(fileEntity)
                                RawFileModelMapper().map(fileEntity)
                            }
                                .subscribeOn(Schedulers.io())
                                .observeOn(AndroidSchedulers.mainThread())
                                .flatMapPublisher {
                                    Flowable.just(
                                        AmityUploadResult.COMPLETE(
                                            AmityImage(it)
                                        )
                                    )
                                }
                        } else {
                            Flowable.just(AmityUploadResult.PROGRESS(AmityUploadInfo(fileProps)))
                        }
                    }
                    .onErrorReturn {
                        val exception = parseErrorResponse(it)
                        deleteFile(file)
                        AmityUploadResult.ERROR(exception)
                    }
            }.onErrorReturn {
                val exception = AmityException.create(it.message, null, AmityError.UNKNOWN)
                AmityUploadResult.ERROR(exception)
            }
    }

    private fun rewriteImageFile(context: Context, fileUri: Uri): Single<File> {
        return getFileAndOrientation(context, fileUri)
                .flatMap { result ->
                    val file = result.first
                    val orientation = result.second
                    val bitmap = BitmapFactory.decodeFile(file.absolutePath).let { bitmap ->
                        orientation?.let { fixOrientation(bitmap, it) } ?: bitmap
                    }
                    saveMediaToCache(context, bitmap)
                }
    }
    
    private fun getInputStream(context: Context, uri: Uri): InputStream? {
        return if (uri.scheme != "content") {
            val file = File(uri.path)
            file.inputStream()
        } else {
            context.contentResolver.run {
                openInputStream(uri)
            }
        }
    }
    
    private fun getFileAndOrientation(context: Context, fileUri: Uri): Single<Pair<File, Int?>> {
        return Single.create { emitter ->
            val file = File(context.cacheDir, "img_cache_${UUID.randomUUID()}.jpg")
            try {
                val exifOrientation: Int? = try {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                        getInputStream(context, fileUri)?.let { inputStream ->
                            val exif = ExifInterface(inputStream).getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL)
                            inputStream.close()
                            exif
                        }
                    } else {
                        ExifInterface(file.absolutePath).getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL)
                    }
                } catch (e: Exception) {
                    null
                }
                getInputStream(context, fileUri)?.let { inputStream ->
                    val outputStream = FileOutputStream(file)
                    var read = 0
                    val maxBufferSize = 1 * 1024 * 1024
                    val bytesAvailable = inputStream.available()
                    val bufferSize = bytesAvailable.coerceAtMost(maxBufferSize)
                    val buffers = ByteArray(bufferSize)
                    while (inputStream.read(buffers).also { read = it } != -1) {
                        outputStream.write(buffers, 0, read)
                    }
                    inputStream.close()
                    outputStream.close()
                    emitter.onSuccess(Pair(file, exifOrientation))
                }
            } catch (e: Throwable) {
                AmityLog.e("Exception ${e.message}")
                emitter.onError(e)
            }
        }
    }

    private fun fixOrientation(bitmap: Bitmap, orientation: Int?): Bitmap {
        return try {
            val matrix = Matrix()
            when (orientation) {
                ExifInterface.ORIENTATION_NORMAL -> return bitmap
                ExifInterface.ORIENTATION_FLIP_HORIZONTAL -> matrix.setScale(-1f, 1f)
                ExifInterface.ORIENTATION_ROTATE_180 -> matrix.setRotate(180f)
                ExifInterface.ORIENTATION_FLIP_VERTICAL -> {
                    matrix.setRotate(180f)
                    matrix.postScale(-1f, 1f)
                }
                ExifInterface.ORIENTATION_TRANSPOSE -> {
                    matrix.setRotate(90f)
                    matrix.postScale(-1f, 1f)
                }
                ExifInterface.ORIENTATION_ROTATE_90 -> matrix.setRotate(90f)
                ExifInterface.ORIENTATION_TRANSVERSE -> {
                    matrix.setRotate(-90f)
                    matrix.postScale(-1f, 1f)
                }
                ExifInterface.ORIENTATION_ROTATE_270 -> matrix.setRotate(-90f)
                else -> return bitmap
            }
            val result =
                Bitmap.createBitmap(bitmap, 0, 0, bitmap.width, bitmap.height, matrix, true)
            bitmap.recycle()
            result
        } catch (e: Exception) {
            AmityLog.e("Fail to rotate image with exception: ${e.message}")
            bitmap
        }
    }

    private fun saveMediaToCache(context: Context, bitmap: Bitmap): Single<File> {
        return Single.create<File> { emitter ->
            try {
                val filename = "img_cache_${UUID.randomUUID()}.jpg"
                val file = if (Build.VERSION.SDK_INT > Build.VERSION_CODES.P) {
                    File("${context.cacheDir.absolutePath}${File.separator}$filename")
                } else {
                    File(context.cacheDir, filename)
                }
                FileOutputStream(file)
                    .apply {
                        use { bitmap.compress(Bitmap.CompressFormat.JPEG, 100, it) }
                        close()
                    }
                emitter.onSuccess(file)
            } catch (e: Throwable) {
                AmityLog.e("error: $e")
                emitter.onError(e)
            }
        }
    }

    private fun deleteFile(file: File) {
        if (file.exists()) {
            val success = file.delete()
        }
    }

    class Builder {

        private lateinit var uri: Uri
        private var uploadId: String? = null
        private var isFullImage = false

        internal fun fileUri(uri: Uri): Builder {
            this.uri = uri
            return this
        }

        fun uploadId(uploadId: String): Builder {
            this.uploadId = uploadId
            return this
        }

        fun isFullImage(isFullImage: Boolean): Builder {
            this.isFullImage = isFullImage
            return this
        }

        fun build(): AmityImageUploadService {
            return AmityImageUploadService(uri, uploadId, isFullImage)
        }
    }
}