package com.amity.socialcloud.sdk.api.social.member.search

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.api.social.member.query.AmityCommunityMembershipSortOption
import com.amity.socialcloud.sdk.model.core.role.AmityRoles
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMember
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMembership
import com.amity.socialcloud.sdk.social.domain.community.member.SearchCommunityMembershipPagingSourceUseCase
import io.reactivex.rxjava3.core.Flowable

class AmityCommunityMemberSearch private constructor(
    private var communityId: String,
    private var roles: AmityRoles,
    private var communityMembership: List<AmityCommunityMembership>?,
    private var sortBy: AmityCommunityMembershipSortOption,
    private var keyword: String,
) {

    /**
     * Search for communities
     * @return LiveCollection of AmityCommunityMember as Flowable<PagingData<AmityCommunityMember>>.
     */
    fun query(): Flowable<PagingData<AmityCommunityMember>> {
        return SearchCommunityMembershipPagingSourceUseCase().execute(
            communityId,
            roles,
            communityMembership,
            sortBy,
            keyword
        )
    }

    class Builder internal constructor(
        private var communityId: String,
        private var keyword: String
    ) {

        private var roles = AmityRoles()
        private var communityMembership: List<AmityCommunityMembership>? = null
        private var sortBy = AmityCommunityMembershipSortOption.DISPLAY_NAME

        /**
         * @param [communityMembership] to filter by the specifying membership value
         * @return the same instance of AmityCommunityMembershipQuery.Builder
         */
        fun membershipFilter(communityMembership: List<AmityCommunityMembership>): Builder {
            return apply { this.communityMembership = communityMembership }
        }

        /**
         * @param [roles] to filter by a list of roleId
         * @return the same instance of AmityCommunityMemberSearch.Builder
         */
        fun roles(roles: List<String>): Builder {
            return apply {
                this.roles = AmityRoles(roles)
            }
        }

        /**
         * @param [sortBy] to specify the sorting option.
         * When [sortBy] is AmityCommunityMembershipSortOption.DISPLAY_NAME, members will be sorted by displayName property ASC.
         * When [sortBy] is AmityCommunityMembershipSortOption.FIRST_CREATED, members will be sorted by createdAt property ASC.
         * When [sortBy] is AmityCommunityMembershipSortOption.LAST_CREATED, members will be sorted by createdAt property DESC.
         * The default value of [sortBy] is AmityCommunityMembershipSortOption.DISPLAY_NAME.
         * @return the same AmityCommunityMembershipQuery.Builder instance.
         */
        fun sortBy(sortBy: AmityCommunityMembershipSortOption): Builder {
            return apply { this.sortBy = sortBy }
        }

        /**
         * Instantiates AmityCommunityMemberSearch with built params.
         * @return new instance of AmityCommunityMemberSearch.
         */
        fun build(): AmityCommunityMemberSearch {
            return AmityCommunityMemberSearch(
                communityId,
                roles,
                communityMembership,
                sortBy,
                keyword,
            )
        }
    }

}