package com.amity.socialcloud.sdk.core.data.analytics

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.core.data.analytics.paging.ViewedUsersMediator
import com.amity.socialcloud.sdk.core.data.session.SessionLocalDataStore
import com.amity.socialcloud.sdk.core.data.user.UserModelMapper
import com.amity.socialcloud.sdk.core.engine.analytics.AnalyticsEventSyncPriority
import com.amity.socialcloud.sdk.model.core.analytics.AmityViewedType
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.ekoapp.ekosdk.internal.api.socket.request.AnalyticsEventRequest
import com.ekoapp.ekosdk.internal.api.socket.request.SendAnalyticsEventsRequest
import com.ekoapp.ekosdk.internal.entity.AnalyticsEventEntity
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime
import org.joda.time.DateTimeZone

@OptIn(ExperimentalPagingApi::class)
class AnalyticsRepository {

    fun saveAnalyticEvent(
        contentId: String,
        contentType: String,
        activityType: String,
        syncPriority: AnalyticsEventSyncPriority
    ): Completable {
        val localDataStore = AnalyticsLocalDataStore()
        return localDataStore.getAnalyticsEventsCount()
            .map { count ->
                if (count >= MAX_ANALYTIC_EVENTS) {
                    localDataStore.deleteOldestEvent()
                        .subscribeOn(Schedulers.io())
                        .subscribe()
                }
            }.map {
                createAnalyticEvent(
                    contentId = contentId,
                    contentType = contentType,
                    activityType = activityType,
                    syncPriority = syncPriority
                )
            }.flatMapCompletable { event ->
                localDataStore.saveAnalyticsEvents(listOf(event))
            }
    }

    fun sendAnalyticsEvents(syncPriority: AnalyticsEventSyncPriority): Completable {
        return AnalyticsLocalDataStore().getAnalyticsEvents(
            userId = AmityCoreClient.getUserId(),
            syncPriority = syncPriority,
            limit = 1000
        ).flatMap { entities ->
            if (entities.isNotEmpty()) {
                val request = createSendAnalyticEventsRequest(entities)
                AnalyticsRemoteDataStore().sendAnalyticsEvents(request)
                    .map { entities }
            } else {
                Single.just(emptyList())
            }
        }.flatMapCompletable { entities ->
            if (entities.isNotEmpty()) {
                AnalyticsLocalDataStore().deleteAnalyticsEvents(entities)
            } else {
                Completable.complete()
            }
        }
    }

    fun deleteAllAnalyticsEvents(): Completable {
        return AnalyticsLocalDataStore().deleteAllAnalyticsEvents()
    }

    fun getViewedUsers(
        viewableItemType: AmityViewedType,
        viewableItemId: String
    ): Flowable<PagingData<AmityUser>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = AmityObjectRepository.DEFAULT_PAGE_SIZE,
                enablePlaceholders = false
            ),
            queryStreamMediator = ViewedUsersMediator(viewableItemType, viewableItemId),
            pagingSourceFactory = { AnalyticsLocalDataStore().getViewedUsersPagingSource(viewableItemType, viewableItemId) },
            modelMapper = UserModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    private fun createAnalyticEvent(
        contentId: String,
        contentType: String,
        activityType: String,
        syncPriority: AnalyticsEventSyncPriority
    ): AnalyticsEventEntity {
        return AnalyticsEventEntity(
            userId = SessionLocalDataStore().getActiveUserId(),
            contentId = contentId,
            contentType = contentType,
            activityType = activityType,
            syncPriority = syncPriority.apiKey
        ).apply {
            val currentDate = DateTime.now(DateTimeZone.UTC)
            this.createdAt = currentDate
            this.updatedAt = currentDate
        }
    }

    private fun createSendAnalyticEventsRequest(entities: List<AnalyticsEventEntity>): SendAnalyticsEventsRequest {
        return SendAnalyticsEventsRequest(
            activities = entities.map { entity ->
                AnalyticsEventRequest(
                    contentId = entity.contentId,
                    contentType = entity.contentType,
                    activityType = entity.activityType,
                    timestamp = entity.createdAt
                )
            }
        )
    }

    companion object {
        private const val MAX_ANALYTIC_EVENTS = 1000
    }
}