package com.amity.socialcloud.sdk.api.social.member.query

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.model.core.role.AmityRoles
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMember
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMembershipFilter
import com.amity.socialcloud.sdk.social.domain.community.member.CommunityMembershipPagingDataUseCase
import io.reactivex.rxjava3.core.Flowable

class AmityCommunityMembershipQuery internal constructor(
    private var communityId: String,
    private var roles: AmityRoles,
    private var filter: AmityCommunityMembershipFilter,
    private var sortBy: AmityCommunityMembershipSortOption
) {

    /**
     * Queries for communities
     * @return LiveCollection of AmityCommunityMember as Flowable<PagingData<AmityCommunityMember>>.
     */
    fun query(): Flowable<PagingData<AmityCommunityMember>> {
        return CommunityMembershipPagingDataUseCase().execute(
            communityId,
            roles,
            filter,
            sortBy
        )
    }

    class Builder(private var communityId: String) {
        private var roles: AmityRoles = AmityRoles()
        private var filter: AmityCommunityMembershipFilter = AmityCommunityMembershipFilter.MEMBER
        private var sortBy = AmityCommunityMembershipSortOption.FIRST_CREATED

        /**
         * @param [filter] to specify AmityCommunityMembershipFilter
         * when [filter] is AmityCommunityMembershipFilter.ALL, the result contains both member and banned users
         * when [filter] is AmityCommunityMembershipFilter.MEMBER, the result excludes banned users
         * when [filter] is AmityCommunityMembershipFilter.BANNED, the result contains only banned users
         * the default value of [filter] is AmityCommunityFilter.MEMBER
         * @return the same instance of AmityCommunityMembershipQuery.Builder
         */
        fun filter(filter: AmityCommunityMembershipFilter): Builder {
            return apply {
                this.filter = filter
            }
        }

        /**
         * @param [roles] to filter by a list of roleId
         * @return the same instance of AmityCommunityMembershipQuery.Builder
         */
        fun roles(roles: List<String>): Builder {
            return apply {
                this.roles = AmityRoles(roles)
            }
        }

        /**
         * @param [sortBy] to specify the sorting option.
         * When [sortBy] is AmityCommunityMembershipSortOption.DISPLAY_NAME, members will be sorted by displayName property ASC.
         * When [sortBy] is AmityCommunityMembershipSortOption.FIRST_CREATED, members will be sorted by createdAt property ASC.
         * When [sortBy] is AmityCommunityMembershipSortOption.LAST_CREATED, members will be sorted by createdAt property DESC.
         * The default value of [sortBy] is AmityCommunityMembershipSortOption.FIRST_CREATED.
         * @return the same AmityCommunityMembershipQuery.Builder instance.
         */
        fun sortBy(sortBy: AmityCommunityMembershipSortOption): Builder {
            return apply { this.sortBy = sortBy }
        }

        /**
         * Instantiates AmityCommunityMembershipQuery with built params.
         * @return new instance of AmityCommunityMembershipQuery.
         */
        fun build(): AmityCommunityMembershipQuery {
            return AmityCommunityMembershipQuery(
                communityId,
                roles,
                filter,
                sortBy
            )
        }
    }

}