package com.amity.socialcloud.sdk.model.social.post

import android.os.Parcelable
import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.api.core.events.AmityTopicSubscription
import com.amity.socialcloud.sdk.api.core.reaction.AmityReactor
import com.amity.socialcloud.sdk.api.social.comment.create.AmityCommentCreationTarget
import com.amity.socialcloud.sdk.api.social.post.flag.AmityPostFlagger
import com.amity.socialcloud.sdk.api.social.post.reaction.AmityPostReactionQuery
import com.amity.socialcloud.sdk.api.social.post.review.AmityReviewStatus
import com.amity.socialcloud.sdk.api.social.post.update.AmityTextPostUpdate
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.helper.core.mention.AmityMentionee
import com.amity.socialcloud.sdk.model.core.events.AmityPostEvents
import com.amity.socialcloud.sdk.model.core.events.AmityTopic
import com.amity.socialcloud.sdk.model.core.file.AmityFile
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.file.AmityVideo
import com.amity.socialcloud.sdk.model.core.file.AmityVideoResolution
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionMap
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionReferenceType
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.amity.socialcloud.sdk.model.social.comment.AmityComment
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMember
import com.amity.socialcloud.sdk.model.social.poll.AmityPoll
import com.amity.socialcloud.sdk.model.video.stream.AmityStream
import com.amity.socialcloud.sdk.socket.util.EkoGson
import com.ekoapp.ekosdk.ReactorObject
import com.ekoapp.ekosdk.internal.usecase.file.GetVideoUrlUseCase
import com.ekoapp.ekosdk.internal.usecase.post.PollGetUseCase
import com.ekoapp.ekosdk.internal.usecase.post.PostDeleteUseCase
import com.amity.socialcloud.sdk.video.domain.stream.GetStreamUseCase
import com.google.common.base.Objects
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import kotlinx.parcelize.Parcelize
import kotlinx.parcelize.TypeParceler
import org.joda.time.DateTime

@Parcelize
@TypeParceler<JsonObject?, JsonObjectParceler>
data class AmityPost internal constructor(
    private val postId: String,
    private val mid: String,
    internal val target: Target,
    private val parentPostId: String?,
    internal val postedUserId: String,
    internal val sharedUserId: String?,
    internal val type: DataType,
    internal val data: Data,
    private val metadata: JsonObject?,
    private val sharedCount: Int,
    internal var isFlaggedByMe: Boolean = false,
    internal var myReactions: List<String> = emptyList(),
    private val reactions: AmityReactionMap,
    private val reactionCount: Int,
    private val commentCount: Int,
    private val flagCount: Int,
    internal var latestComments: List<AmityComment> = emptyList(),
    internal val childrenPostIds: List<String>,
    internal var children: List<AmityPost> = emptyList(),
    internal var postedUser: AmityUser? = null,
    internal var sharedUser: AmityUser? = null,
    private val isDeleted: Boolean,
    private val feedType: AmityFeedType,
    internal var mentionees: List<AmityMentionee> = emptyList(),
    private val editedAt: DateTime?,
    private val createdAt: DateTime?,
    private val updatedAt: DateTime?,
    internal val path: String,
    private val impression: Int,
    private val reach: Int,
) : Parcelable, ReactorObject {

    /*
        Using Parcelize with sealed class, properties must be in the constructors of subclasses.
     */
    sealed class Data : Parcelable {

        @Parcelize
        class TEXT(
            internal val postId: String,
            internal val text: String
        ) : Data() {

            fun getPostId(): String {
                return postId
            }

            fun getText(): String {
                return text
            }

            @Deprecated("Use AmitySocialClient.newPostRepository().updatePost() instead")
            fun edit(): AmityTextPostUpdate.Builder {
                return AmityTextPostUpdate.Builder(getPostId())
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is TEXT
                        && Objects.equal(other.postId, postId)
                        && Objects.equal(other.text, text))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(postId, text)
            }

        }

        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class IMAGE(
            internal val postId: String,
            internal val fileId: String,
            internal val rawData: JsonObject?,
            internal var image: AmityImage? = null
        ) : Data() {

            fun getPostId(): String {
                return postId
            }

            fun getImage(): AmityImage? {
                return image
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is IMAGE
                        && Objects.equal(other.postId, postId)
                        && Objects.equal(other.rawData, rawData))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(postId, rawData)
            }

        }

        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class FILE(
            internal val postId: String,
            internal val fileId: String,
            internal val rawData: JsonObject?,
            internal var file: AmityFile? = null
        ) : Data() {

            fun getPostId(): String {
                return postId
            }

            fun getFile(): AmityFile? {
                return file
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is FILE
                        && Objects.equal(other.postId, postId)
                        && Objects.equal(other.rawData, rawData))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(postId, rawData)
            }

        }

        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class VIDEO(
            internal val postId: String,
            internal val thumbnailFileId: String,
            internal val rawData: JsonObject?,
            internal var thumbnail: AmityImage? = null
        ) : Data() {


            fun getPostId(): String {
                return postId
            }

            fun getThumbnailImage(): AmityImage? {
                return thumbnail
            }

            fun getVideo(quality: AmityVideo.Quality? = null): Single<AmityVideo> {
                return GetVideoUrlUseCase().execute(rawData, quality)
            }

            fun getAvailableResolutions(): Single<List<AmityVideoResolution>> {
                return getVideo()
                    .map {
                        it.getResolutions()
                    }
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is VIDEO
                        && Objects.equal(other.postId, postId)
                        && Objects.equal(other.rawData, rawData))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(postId, rawData)
            }

        }

        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class LIVE_STREAM(
            internal val postId: String,
            internal val streamId: String,
            internal val rawData: JsonObject?
        ) : Data() {

            fun getPostId(): String {
                return postId
            }

            fun getStream(): Flowable<AmityStream> {
                return GetStreamUseCase().execute(streamId)
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is LIVE_STREAM
                        && Objects.equal(other.postId, postId)
                        && Objects.equal(other.rawData, rawData))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(postId, rawData)
            }

        }

        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class POLL(
            internal val postId: String,
            internal val pollId: String,
            internal val rawData: JsonObject?
        ) : Data() {

            fun getPostId(): String {
                return postId
            }

            fun getPollId(): String {
                return pollId
            }

            fun getPoll(): Flowable<AmityPoll> {
                return PollGetUseCase().execute(pollId)
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is POLL
                        && Objects.equal(other.postId, postId)
                        && Objects.equal(other.rawData, rawData))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(postId, rawData)
            }

        }


        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class CUSTOM(
            internal val postId: String,
            internal val type: DataType.CUSTOM,
            internal val rawData: JsonObject?
        ) : Data() {

            fun getPostId(): String {
                return postId
            }

            fun getDataType(): String = type.customTypeName

            fun getRawData(): JsonObject? = rawData

            fun <T> getSerializedData(clazz: Class<T>): T? {
                return if (rawData == null) {
                    null
                } else {
                    EkoGson.get().fromJson(rawData, clazz)
                }
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is CUSTOM
                        && Objects.equal(other.postId, postId)
                        && Objects.equal(other.rawData, rawData))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(postId, rawData)
            }

        }

    }

    sealed class DataType : Parcelable {
        abstract fun getApiKey(): String
        abstract fun getFileIdKey(): String

        @Parcelize
        object TEXT : DataType() {
            override fun getApiKey(): String = "text"
            override fun getFileIdKey(): String = ""
        }

        @Parcelize
        object IMAGE : DataType() {
            override fun getApiKey(): String = "image"
            override fun getFileIdKey(): String = "fileId"
        }

        @Parcelize
        object FILE : DataType() {
            override fun getApiKey(): String = "file"
            override fun getFileIdKey(): String = "fileId"
        }

        @Parcelize
        object LIVE_STREAM : DataType() {
            override fun getApiKey(): String = "liveStream"
            override fun getFileIdKey(): String = "fileId"
        }

        @Parcelize
        object POLL : DataType() {
            override fun getApiKey(): String = "poll"
            override fun getFileIdKey(): String = ""
        }

        @Parcelize
        object VIDEO : DataType() {
            override fun getApiKey(): String = "video"
            override fun getFileIdKey(): String = "thumbnailFileId"
        }

        @Parcelize
        class CUSTOM(val customTypeName: String) : DataType() {
            override fun getApiKey(): String = customTypeName
            override fun getFileIdKey(): String = ""
        }

        companion object {
            fun isAttachmentType(apiKey: String?): Boolean {
                return apiKey == IMAGE.getApiKey()
                        || apiKey == VIDEO.getApiKey()
                        || apiKey == FILE.getApiKey()
            }

            fun sealedOf(dataType: String?): DataType {
                return when (dataType) {
                    TEXT.getApiKey() -> TEXT
                    IMAGE.getApiKey() -> IMAGE
                    FILE.getApiKey() -> FILE
                    VIDEO.getApiKey() -> VIDEO
                    LIVE_STREAM.getApiKey() -> LIVE_STREAM
                    POLL.getApiKey() -> POLL
                    else -> CUSTOM(dataType ?: "")
                }
            }
        }
    }

    internal enum class TargetType(val apiKey: String) {
        USER("user"),
        COMMUNITY("community");

        companion object {
            fun enumOf(value: String?): TargetType {
                return values().find { it.apiKey == value } ?: USER
            }
        }
    }

    sealed class Target(
        private val targetType: TargetType,
        private val targetId: String
    ) : Parcelable {

        internal fun getTargetType(): TargetType {
            return targetType
        }

        internal fun getTargetId(): String {
            return targetId
        }

        @Parcelize
        class COMMUNITY internal constructor(
            internal val targetCommunityId: String,
            internal val postedCommunityMemberId: String,
            internal var targetCommunity: AmityCommunity? = null,
            internal var postedCommunityMember: AmityCommunityMember? = null
        ) : Target(TargetType.COMMUNITY, targetCommunityId) {

            fun getCommunityId(): String {
                return targetCommunityId
            }

            fun getCommunity(): AmityCommunity? {
                return targetCommunity
            }

            fun getCreatorMember(): AmityCommunityMember? {
                return postedCommunityMember
            }

            companion object {
                fun create(communityId: String): COMMUNITY {
                    return COMMUNITY(communityId, AmityCoreClient.getUserId())
                }
            }

        }

        @Parcelize
        class USER internal constructor(
            internal val targetUserId: String,
            internal var targetUser: AmityUser? = null
        ) : Target(TargetType.USER, targetUserId) {

            fun getUserId(): String {
                return targetUserId
            }

            fun getUser(): AmityUser? {
                return targetUser
            }

            companion object {
                fun create(userId: String): USER {
                    return USER(userId)
                }
            }
        }

        @Parcelize
        object UNKNOWN : Target(TargetType.USER, "")
    }

    fun getPostId(): String {
        return postId
    }

    fun getTarget(): Target {
        return target
    }

    fun getParentPostId(): String? {
        return parentPostId
    }

    fun getType(): DataType {
        return type
    }

    fun getData(): Data {
        return data
    }

    fun getReactionMap(): AmityReactionMap {
        return reactions
    }

    fun getReactionCount(): Int {
        return reactionCount
    }

    fun getCommentCount(): Int {
        return commentCount
    }

    fun getFlagCount(): Int {
        return flagCount
    }

    fun getMyReactions(): List<String> {
        return myReactions
    }

    fun getLatestComments(): List<AmityComment> {
        return latestComments
    }

    fun getChildren(): List<AmityPost> {
        return children
    }

    fun getCreator(): AmityUser? {
        return postedUser
    }

    fun isEdited(): Boolean {
        return editedAt?.isAfter(createdAt) ?: false
    }

    fun getEditedAt(): DateTime? {
        return editedAt
    }

    fun getCreatedAt(): DateTime? {
        return createdAt
    }

    fun getUpdatedAt(): DateTime? {
        return updatedAt
    }

    fun isFlaggedByMe(): Boolean {
        return isFlaggedByMe
    }

    fun isDeleted(): Boolean {
        return isDeleted
    }

    fun getCreatorId(): String {
        return postedUserId
    }

    @Deprecated("Use getReviewStatus() instead")
    fun getFeedType(): AmityFeedType {
        return feedType
    }

    fun getReviewStatus(): AmityReviewStatus {
        return AmityReviewStatus.enumOf(feedType.apiKey)
    }

    fun getMetadata(): JsonObject? {
        return metadata
    }

    fun getMentionees(): List<AmityMentionee> {
        return mentionees
    }

    fun react(): AmityReactor {
        return AmityReactor(AmityReactionReferenceType.POST, postId)
    }

    fun comment(): AmityCommentCreationTarget.Builder {
        return AmityCommentCreationTarget().post(getPostId())
    }

    @Deprecated("Use AmitySocialClient.newPostRepository() instead")
    fun report(): AmityPostFlagger {
        return AmityPostFlagger(postId)
    }

    @Deprecated("Use AmitySocialClient.newPostRepository().softDelete() instead")
    fun delete(): Completable {
        return delete(false)
    }

    @Deprecated("Use AmitySocialClient.newPostRepository().softDelete() or hardDelete() instead")
    fun delete(hardDelete: Boolean): Completable {
        return PostDeleteUseCase().execute(postId, hardDelete)
    }

    fun getReactions(): AmityPostReactionQuery.Builder {
        return AmityPostReactionQuery.Builder(getPostId())
    }

    fun subscription(events: AmityPostEvents): AmityTopicSubscription {
        return AmityTopicSubscription(AmityTopic.POST(this, events))
    }

    fun getImpression(): Int {
        return impression
    }

    fun getReach(): Int {
        return reach
    }

    fun analytics(): AmityPostAnalytics {
        return AmityPostAnalytics(mid)
    }

    override fun updatedAt(): DateTime? {
        return updatedAt
    }

    override fun uniqueId(): String {
        return postId
    }
}

