package com.gitee.aachen0.util;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.Properties;

public class JDBCUtil {
    /**
     * 从java属性文件中的配置建立一个jdbc连接
     *
     * @param jdbc jdbc配置文件,示例如下：
     *             driverName=com.mysql.jdbc.Driver
     *             url=jdbc:mysql://localhost:3306/yourdatabase
     *             user=root
     *             password=root
     * @return jdbc连接Connection
     */
    
    public static Connection getConnection(String jdbc) {
        InputStream is = JDBCUtil.class.getResourceAsStream(jdbc);
        Properties pro = new Properties();
        Connection connection = null;
        try {
            pro.load(is);
            Class.forName(pro.getProperty("driverName"));
            connection = DriverManager.getConnection(pro.getProperty("url"), pro.getProperty("user"),
                    pro.getProperty("password"));
        } catch (IOException | SQLException | ClassNotFoundException e) {
            e.printStackTrace();
        }
        return connection;
    }
    
    /**
     * 将结果集转换成对应的实体类对象
     *
     * @param rs    结果集
     * @param clazz 实体类的Class对象
     * @return 实体类对象
     */
    public static <T> Object rsToObject(ResultSet rs, Class<T> clazz) {
        Object object = null;
        try {
            object = clazz.newInstance();
            ResultSetMetaData data = rs.getMetaData();
            int n = data.getColumnCount();
            for (int i = 1; i <= n; i++) {
                String columnName = data.getColumnName(i);
                String columnType = data.getColumnTypeName(i);
                Object o = rs.getObject(columnName);
                String setter = "set" + columnName.substring(0, 1).toUpperCase() + columnName.substring(1);// 拼出set的方法名
                Class paraType = jdbcToJavaType(columnType);
                Method method = clazz.getMethod(setter, paraType);
                Class packType = toPackType(paraType);
                method.invoke(object, packType.cast(o));
            }
        } catch (InstantiationException | InvocationTargetException | NoSuchMethodException | IllegalAccessException | SQLException e) {
            e.printStackTrace();
        }
        return object;
    }
    
    /**
     * 将基本类型的类转换为对应的包装类型
     *
     * @param paraType 基本数据类型
     * @return 包装数据类型
     */
    private static Class toPackType(Class paraType) {
        switch (paraType.getName()) {
            case "int":
                return Integer.class;
            case "long":
                return Long.class;
            case "byte":
                return Byte.class;
            case "short":
                return Short.class;
            case "float":
                return Float.class;
            case "double":
                return Double.class;
        }
        return paraType;
    }
    
    /**
     * 将jdbc的数据类型映射到java基本数据类型
     *
     * @param columnType 结果集中取出的字段数据类型
     * @return 对应的java数据类
     */
    private static Class jdbcToJavaType(String columnType) {
        
        switch (columnType) {
            case "TINYINT":
            case "TINYINT UNSIGNED":
            case "SMALLINT":
            case "SMALLINT UNSIGNED":
            case "MEDIUMINT":
            case "MEDIUMINT UNSIGNED":
            case "INT":
                return int.class;
            case "BIGINT":
            case "INT UNSIGNED":
                return long.class;
            case "BIGINT UNSIGNED":
                return BigInteger.class;
            case "FLOAT":
            case "FLOAT UNSIGNED":
                return float.class;
            case "DOUBLE":
            case "DOUBLE UNSIGNED":
                return double.class;
            case "CHAR":
            case "VARCHAR":
                return String.class;
            case "DECIMAL":
            case "DECIMAL UNSIGNED":
                return BigDecimal.class;
            
            case "TIME":
                return java.sql.Time.class;
            case "YEAR":
            case "DATE":
                return java.sql.Date.class;
            case "DATETIME":
            case "TIMESTAMP":
                return java.sql.Timestamp.class;
        }
        return String.class;
    }
}
