package com.tm.datamanager.preferencesmanager;

import android.content.Context;
import android.content.SharedPreferences;

/**
 * Created by Navas on 19/06/15
 * This class control all information saved in the preference files
 */
public class PreferencesManager {

    private Context context;

    private final SharedPreferences sharedPreferences;

    private PreferencesConfigurations configuration;

    KeystoreManager keystoreManager;

    public PreferencesManager(Context context, PreferencesConfigurations configuration){
        this.context = context;
        this.configuration = configuration;
        keystoreManager = new KeystoreManager(context, configuration.getPreferencesFileName());
        sharedPreferences = context.getSharedPreferences(configuration.getPreferencesFileName(), Context.MODE_PRIVATE);
    }

    //Integer type preference methods

    /**
     * This method sets a value to a preference of the integer type
     *
     * @param   value       The value to be set
     * @param   preference  The key that identifies the preference
     * @return              A boolean value that indicates if the preference has been set
     * @see                 SharedPreferences
     */
    public boolean setValue(int value, String preference) {
        return sharedPreferences.edit().putInt(preference, value).commit();
    }

    /**
     * This method gets a value from a preference of the integer type
     *
     * @param   preference  The key that identifies the preference
     * @return              The value of the preference or the default value defined
     *                      in the configurations
     * @see                 SharedPreferences
     */
    public int getIntValue(String preference) {
        return sharedPreferences.getInt(preference,
                configuration.getIntPreferenceDefaultValue());
    }

    public int getIntValue(String preference, int defaultValue) {
        return sharedPreferences.getInt(preference,
                defaultValue);
    }

    //Boolean type preference methods

    /**
     * This method sets a value to a preference of the boolean type
     *
     * @param   value       The value to be set
     * @param   preference  The key that identifies the preference
     * @return              A boolean value that indicates if the preference has been set
     * @see                 SharedPreferences
     */
    public boolean setValue(boolean value, String preference) {
        return sharedPreferences.edit().putBoolean(preference, value).commit();
    }

    /**
     * This method gets a value from a preference of the boolean type
     *
     * @param   preference  The key that identifies the preference
     * @return              The value of the preference or the default value defined
     *                      in the configurations
     * @see                 SharedPreferences
     */
    public boolean getBooleanValue(String preference) {
        return sharedPreferences.getBoolean(preference,
                configuration.getBooleanPreferenceDefaultValue());
    }

    public boolean getBooleanValue(String preference, boolean defaultValue) {
        return sharedPreferences.getBoolean(preference,
                defaultValue);
    }

    //String type preference methods

    /**
     * This method sets a value to a preference of the String type
     *
     * @param   value       The value to be set
     * @param   preference  The key that identifies the preference
     * @return              A boolean value that indicates if the preference has been set
     * @see                 SharedPreferences
     */
    public boolean setValue(String value, String preference) {
        return sharedPreferences.edit().putString(preference, value).commit();
    }

        /**
     * This method sets an encrypt value to a preference of the String type
     *
     * @param   value       The value to be set
     * @param   preference  The key that identifies the preference
     * @return              A boolean value that indicates if the preference has been set
     * @see                 SharedPreferences
     */
    public boolean setEncryptValue(String value, String preference) {
        if(value.length() > 245){
            if(contains(preference+1)) deleteEncryptedPrefences(preference);
            int splits = value.length()/ 245;
            for (int i = 1; i <= splits; i++){
                saveEncryptPreference(value.substring((i-1)*245, i *245),preference+i);
            }
            if(value.length() % 245 != 0){
                saveEncryptPreference(value.substring(splits++ * 245),preference+splits);
            }
            saveEncryptPreference(splits+"", preference);
            return true;
        }else return saveEncryptPreference(value, preference);
    }

    private boolean saveEncryptPreference(String value, String preference) {
        value = keystoreManager.encryptText(value);
        return sharedPreferences.edit().putString(preference, value).commit();
    }

    /**
     * This method gets an encrypt value from a preference of the String type
     *
     * @param   preference  The key that identifies the preference
     * @return              The value of the preference or the default value defined
     *                      in the configurations
     * @see                 SharedPreferences
     */
    public String getExcryptValue(String preference) {
        if(contains(preference+1)){
            String value = "";
            int splits = Integer.parseInt(getEncryptPreference(preference));
            for (int i = 1; i <= splits; i++){
                value+=getEncryptPreference(preference+i);
            }
            return value;
        }else return getEncryptPreference(preference);
    }

    private String getEncryptPreference(String preference) {
        String value = sharedPreferences.getString(preference,
                configuration.getStringPreferenceDefaultValue());
        return keystoreManager.decryptText(value);
    }

    public boolean deleteEncryptedPrefences(String key){
        int splits = Integer.parseInt(getEncryptPreference(key));
        for (int i = 1; i <= splits; i++){
            removePreference(key+i);
        }
        return removePreference(key);
    }


    /**
     * This method gets a value from a preference of the String type
     *
     * @param   preference  The key that identifies the preference
     * @return              The value of the preference or the default value defined
     *                      in the configurations
     * @see                 SharedPreferences
     */
    public String getStringValue(String preference) {
        return sharedPreferences.getString(preference,
                configuration.getStringPreferenceDefaultValue());
    }

    public String getStringValue(String preference, String defaultValue) {
        return sharedPreferences.getString(preference,
                defaultValue);
    }

    //Other methods

    /**
     * This method deletes a preference from the preferences file
     *
     * @param   preference  The key that identifies the preference
     * @return              A boolean value that indicates if the preference has been deleted
     * @see                 SharedPreferences
     */
    public boolean removePreference(String preference){
        if(contains(preference+1)) return deleteEncryptedPrefences(preference);
        return sharedPreferences.edit().remove(preference).commit();
    }

    /**
     * This method deletes all preferences from the preferences file
     *
     * @return              A boolean value that indicates if the preferences have been deleted
     * @see                 SharedPreferences
     */
    public boolean clearPreferences(){
        return sharedPreferences.edit().clear().commit();
    }

    public boolean contains(String key){
        return sharedPreferences.contains(key);
    }

}
