package com.tm.datamanager.preferencesmanager;

import android.content.Context;
import android.content.SharedPreferences;

/**
 * Created by Navas on 19/06/15
 * This class control all information saved in the preference files
 */
public class PreferencesManager {

    private Context context;

    private final SharedPreferences sharedPreferences;

    private PreferencesConfigurations configuration;

    public PreferencesManager(Context context, PreferencesConfigurations configuration){
        this.context = context;
        this.configuration = configuration;
        sharedPreferences = context.getSharedPreferences(configuration.getPreferencesFileName(), Context.MODE_PRIVATE);
    }

    //Integer type preference methods

    /**
     * This method sets a value to a preference of the integer type
     *
     * @param   value       The value to be set
     * @param   preference  The key that identifies the preference
     * @return              A boolean value that indicates if the preference has been set
     * @see                 SharedPreferences
     */
    public boolean setValue(int value, String preference) {
        return sharedPreferences.edit().putInt(preference, value).commit();
    }

    /**
     * This method gets a value from a preference of the integer type
     *
     * @param   preference  The key that identifies the preference
     * @return              The value of the preference or the default value defined
     *                      in the configurations
     * @see                 SharedPreferences
     */
    public int getIntValue(String preference) {
        return sharedPreferences.getInt(preference,
                configuration.getIntPreferenceDefaultValue());
    }

    public int getIntValue(String preference, int defaultValue) {
        return sharedPreferences.getInt(preference,
                defaultValue);
    }

    //Boolean type preference methods

    /**
     * This method sets a value to a preference of the boolean type
     *
     * @param   value       The value to be set
     * @param   preference  The key that identifies the preference
     * @return              A boolean value that indicates if the preference has been set
     * @see                 SharedPreferences
     */
    public boolean setValue(boolean value, String preference) {
        return sharedPreferences.edit().putBoolean(preference, value).commit();
    }

    /**
     * This method gets a value from a preference of the boolean type
     *
     * @param   preference  The key that identifies the preference
     * @return              The value of the preference or the default value defined
     *                      in the configurations
     * @see                 SharedPreferences
     */
    public boolean getBooleanValue(String preference) {
        return sharedPreferences.getBoolean(preference,
                configuration.getBooleanPreferenceDefaultValue());
    }

    public boolean getBooleanValue(String preference, boolean defaultValue) {
        return sharedPreferences.getBoolean(preference,
                defaultValue);
    }

    //String type preference methods

    /**
     * This method sets a value to a preference of the String type
     *
     * @param   value       The value to be set
     * @param   preference  The key that identifies the preference
     * @return              A boolean value that indicates if the preference has been set
     * @see                 SharedPreferences
     */
    public boolean setValue(String value, String preference) {
        return sharedPreferences.edit().putString(preference, value).commit();
    }

    /**
     * This method gets a value from a preference of the String type
     *
     * @param   preference  The key that identifies the preference
     * @return              The value of the preference or the default value defined
     *                      in the configurations
     * @see                 SharedPreferences
     */
    public String getStringValue(String preference) {
        return sharedPreferences.getString(preference,
                configuration.getStringPreferenceDefaultValue());
    }

    public String getStringValue(String preference, String defaultValue) {
        return sharedPreferences.getString(preference,
                defaultValue);
    }

    //Other methods

    /**
     * This method deletes a preference from the preferences file
     *
     * @param   preference  The key that identifies the preference
     * @return              A boolean value that indicates if the preference has been deleted
     * @see                 SharedPreferences
     */
    public boolean removePreference(String preference){
        return sharedPreferences.edit().remove(preference).commit();
    }

    /**
     * This method deletes all preferences from the preferences file
     *
     * @return              A boolean value that indicates if the preferences have been deleted
     * @see                 SharedPreferences
     */
    public boolean clearPreferences(){
        return sharedPreferences.edit().clear().commit();
    }

}
