package com.tm.datamanager.webservicesmanager;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.widget.ArrayAdapter;
import android.widget.Toast;

import com.android.volley.DefaultRetryPolicy;
import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.toolbox.Volley;
import com.tm.datamanager.DataManager;
import com.tm.datamanager.preferencesmanager.PreferencesConfigurations;
import com.tm.datamanager.webservicesmanager.configurations.DummyResponses;
import com.tm.datamanager.webservicesmanager.configurations.WebServicesConfiguration;
import com.tm.datamanager.webservicesmanager.requests.CustomRequest;
import com.tm.datamanager.webservicesmanager.requests.CustomRequestParallelList;
import com.tm.datamanager.webservicesmanager.requests.DummyRequestQueue;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;


/**
 * Created by Navas on 19/06/15
 * This class manage the web call services.
 */
public class WebServicesManager implements RequestQueue.RequestFinishedListener{

    private Context context;

    // This is the principal request queue, only there will be an only instance of this queue
    private RequestQueue requestQueue;


    private LinkedList<Request> queue = new LinkedList<>();

    // This list contains the call that will be called in parallel
    private ArrayList<RequestQueue> parallelRequestQueues = new ArrayList<RequestQueue>();

    // Session
    public static Map<String, String> headers;

    public static WebServicesConfiguration configuration;

    public WebServicesManager(Context context, WebServicesConfiguration configuration, HashMap<String, String> dummyResponses){
        this.context = context;
        this.configuration = configuration;
        if(dummyResponses != null)
            DummyResponses.DUMMY_RESPONSES = dummyResponses;
        headers = null;
        requestQueue = generateRequestQueue();
        if(configuration.isDUMMY()){
            configuration.setEnvironment("");
            configuration.setEnvironmentName("DUMMY");
        }else
        // Load the last environment selected
        configuration.setEnvironment(configuration.getENVIRONMENTS()
                [DataManager.getPreferencesManagerInstance().getIntValue("ENVIRONMENT")]);
        if(configuration.isSHOW_REQUEST_LOGS_FILE())
            generateLogFile();
    }

    private void generateLogFile(){
        File root = android.os.Environment.getExternalStorageDirectory();
        File log = new File(root.getAbsolutePath() + "/" + configuration.getFILE_LOG());
        try {
            FileWriter pw = new FileWriter(log, false);
            pw.append("");
            pw.flush();
            pw.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * It check if the app has enabled dummy mode to return a custom requestqueue that return dummy
     * responses
     * @return
     */
    private RequestQueue generateRequestQueue(){
        if(configuration.isDUMMY())
            return new DummyRequestQueue();
        else
            return Volley.newRequestQueue(context.getApplicationContext());
    }

    /**
     * When a Request finishes, it throws the next request in the queue
     */
    @Override
    public void onRequestFinished(Request request) {
        if(queue.size() != 0) requestQueue.add(queue.removeFirst());
    }

    /**
     * Add a request to the principal request queue
     * @param request request for add to the queue
     * @param tag tag to identify the request
     */
    public void addRequest(Request request, String tag){
        request.setTag(tag);
        configureRetryPolicy(request);
        if(queue.size() == 0) requestQueue.add(request);
        else queue.addLast(request);

    }

    /**
     * When you add a parallel request, this will not be added to the principal queue, by these reason
     * all parallel request that will be added will launched in other threads
     * @param request request for add to the queue
     * @param tag tag to identify the request
     */
    public void addParallelRequest(Request request, String tag){
        final RequestQueue requestQueue = generateRequestQueue();
        request.setTag(tag);
        configureRetryPolicy(request);
        requestQueue.add(request);
        requestQueue.addRequestFinishedListener(new RequestQueue.RequestFinishedListener<Object>() {
            @Override
            public void onRequestFinished(Request<Object> request) {
                parallelRequestQueues.remove(requestQueue);
            }
        });
        parallelRequestQueues.add(requestQueue);
    }

    public void addParallelRequestList(final CustomRequestParallelList requestList, String tag){
        for(CustomRequest request : requestList) {
            final RequestQueue requestQueue = generateRequestQueue();
            request.setTag(tag);
            configureRetryPolicy(request);
            requestQueue.add(request);
            requestQueue.addRequestFinishedListener(new RequestQueue.RequestFinishedListener<Object>() {
                @Override
                public void onRequestFinished(Request<Object> request) {
                    parallelRequestQueues.remove(requestQueue);
                    requestList.finishRequest();
                }
            });
            parallelRequestQueues.add(requestQueue);
        }
    }

    /**
     * It set default RetryPolicy to the request if it hasn't
     * @param request
     */
    private void configureRetryPolicy(Request request){
        if(request.getRetryPolicy() == null){
            request.setRetryPolicy(new DefaultRetryPolicy(configuration.getTIMEOUT(),
                    configuration.getATTEMPTS(),
                    configuration.getBACKOFFMULTIPLIER()));
        }
    }

    /**
     * Cancel the request by his tag
     * @param tag to identify the request on the queue
     */
    public void cancelRequest(String tag){
        requestQueue.cancelAll(tag);
        for(RequestQueue requestQueue : parallelRequestQueues)
            requestQueue.cancelAll(tag);
        for(Request request : queue)
            if(request.getTag().equals(tag)){
                queue.remove(request);
                break;
            }
    }

    /**
     * Show a dialog for select the environment
     * @param context
     */
    public void configureEnvironment(final Context context){
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setTitle("ENVIRONMENTS");
        final ArrayAdapter<String> adapter = new ArrayAdapter<String>(context, android.R.layout.select_dialog_singlechoice,configuration.getENVIRONMENTS_NAMES());
        builder.setAdapter(adapter, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialogInterface, int i) {
                configuration.setEnvironment(configuration.getENVIRONMENTS()[i]);
                configuration.setEnvironmentName(configuration.getENVIRONMENTS_NAMES()[i]);
                DataManager.getPreferencesManagerInstance().setValue(i, "ENVIRONMENT");
                Toast.makeText(context, configuration.getEnvironment(), Toast.LENGTH_SHORT).show();
            }
        });
        builder.create().show();
    }

}
