/*
 * Decompiled with CFR 0.152.
 */
package com.gfycat.disklrucache;

import com.gfycat.common.Action1;
import com.gfycat.disklrucache.RandomAccessOutput;
import com.gfycat.disklrucache.StrictLineReader;
import com.gfycat.disklrucache.Util;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.io.IOUtils;

public final class DiskLruCache
implements Closeable {
    private static Action1<Throwable> assertionLogger;
    static final String JOURNAL_FILE = "journal";
    static final String JOURNAL_FILE_TEMP = "journal.tmp";
    static final String JOURNAL_FILE_BACKUP = "journal.bkp";
    static final String MAGIC = "libcore.io.DiskLruCache";
    static final String VERSION_1 = "1";
    static final long ANY_SEQUENCE_NUMBER = -1L;
    static final String STRING_KEY_PATTERN = "[\\.a-z0-9_-]{1,120}";
    static final Pattern LEGAL_KEY_PATTERN;
    private static final String CLEAN = "CLEAN";
    private static final String DIRTY = "DIRTY";
    private static final String REMOVE = "REMOVE";
    private static final String READ = "READ";
    private final File directory;
    private final File journalFile;
    private final File journalFileTmp;
    private final File journalFileBackup;
    private final int appVersion;
    private long maxSize;
    private long size = 0L;
    private Writer journalWriter;
    private final LinkedHashMap<String, Entry> lruEntries = new LinkedHashMap(0, 0.75f, true);
    private int redundantOpCount;
    private long nextSequenceNumber = 0L;
    final ThreadPoolExecutor executorService = new ThreadPoolExecutor(0, 1, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>());
    private final Callable<Void> cleanupCallable = new Callable<Void>(){

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Void call() throws Exception {
            DiskLruCache diskLruCache = DiskLruCache.this;
            synchronized (diskLruCache) {
                if (DiskLruCache.this.journalWriter == null) {
                    return null;
                }
                DiskLruCache.this.trimToSize();
                if (DiskLruCache.this.journalRebuildRequired()) {
                    DiskLruCache.this.rebuildJournal();
                    DiskLruCache.this.redundantOpCount = 0;
                }
            }
            return null;
        }
    };
    private static final OutputStream NULL_OUTPUT_STREAM;

    public static void setupAssertionsLogger(Action1<Throwable> assertionLogger) {
        DiskLruCache.assertionLogger = assertionLogger;
    }

    private static void logAssertion(Throwable throwable) {
        if (assertionLogger != null) {
            assertionLogger.call((Object)throwable);
        }
    }

    private DiskLruCache(File directory, int appVersion, long maxSize) {
        this.directory = directory;
        this.appVersion = appVersion;
        this.journalFile = new File(directory, JOURNAL_FILE);
        this.journalFileTmp = new File(directory, JOURNAL_FILE_TEMP);
        this.journalFileBackup = new File(directory, JOURNAL_FILE_BACKUP);
        this.maxSize = maxSize;
    }

    public static DiskLruCache open(File directory, int appVersion, long maxSize) throws IOException {
        if (maxSize <= 0L) {
            throw new IllegalArgumentException("maxSize <= 0");
        }
        File backupFile = new File(directory, JOURNAL_FILE_BACKUP);
        if (backupFile.exists()) {
            File journalFile = new File(directory, JOURNAL_FILE);
            if (journalFile.exists()) {
                backupFile.delete();
            } else {
                DiskLruCache.renameTo(backupFile, journalFile, false);
            }
        }
        DiskLruCache cache = new DiskLruCache(directory, appVersion, maxSize);
        if (cache.journalFile.exists()) {
            try {
                cache.readJournal();
                cache.processJournal();
                cache.cleanupDirectory();
                return cache;
            }
            catch (IOException journalIsCorrupt) {
                DiskLruCache.logAssertion(new IllegalStateException("DiskLruCache " + directory + " is corrupt: " + journalIsCorrupt.getMessage() + ", removing"));
                cache.delete();
            }
        }
        directory.mkdirs();
        cache = new DiskLruCache(directory, appVersion, maxSize);
        cache.rebuildJournal();
        return cache;
    }

    private void cleanupDirectory() {
        File[] files = this.directory.listFiles();
        if (files == null) {
            return;
        }
        for (File file : files) {
            if (this.lruEntries.containsValue(file.getName())) continue;
            file.delete();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readJournal() throws IOException {
        StrictLineReader reader = new StrictLineReader(new FileInputStream(this.journalFile), Util.US_ASCII);
        try {
            String magic = reader.readLine();
            String version = reader.readLine();
            String appVersionString = reader.readLine();
            String blank = reader.readLine();
            if (!(MAGIC.equals(magic) && VERSION_1.equals(version) && Integer.toString(this.appVersion).equals(appVersionString) && "".equals(blank))) {
                throw new IOException("unexpected journal header: [" + magic + ", " + version + ", " + blank + "]");
            }
            int lineCount = 0;
            try {
                while (true) {
                    this.readJournalLine(reader.readLine());
                    ++lineCount;
                }
            }
            catch (EOFException endOfJournal) {
                this.redundantOpCount = lineCount - this.lruEntries.size();
                if (reader.hasUnterminatedLine()) {
                    this.rebuildJournal();
                } else {
                    this.journalWriter = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(this.journalFile, true), Util.US_ASCII));
                }
                Util.closeQuietly(reader);
            }
        }
        catch (Throwable throwable) {
            Util.closeQuietly(reader);
            throw throwable;
        }
    }

    private void readJournalLine(String line) throws IOException {
        Entry entry;
        String key;
        int firstSpace = line.indexOf(32);
        if (firstSpace == -1) {
            throw new IOException("unexpected journal line: " + line);
        }
        int keyBegin = firstSpace + 1;
        int secondSpace = line.indexOf(32, keyBegin);
        if (secondSpace == -1) {
            key = line.substring(keyBegin);
            if (firstSpace == REMOVE.length() && line.startsWith(REMOVE)) {
                this.lruEntries.remove(key);
                return;
            }
        } else {
            key = line.substring(keyBegin, secondSpace);
        }
        if ((entry = this.lruEntries.get(key)) == null) {
            entry = new Entry(key);
            this.lruEntries.put(key, entry);
        }
        if (secondSpace != -1 && firstSpace == CLEAN.length() && line.startsWith(CLEAN)) {
            String[] parts = line.substring(secondSpace + 1).split(" ");
            entry.readable = true;
            entry.currentEditor = null;
            entry.setLengths(parts);
        } else if (secondSpace == -1 && firstSpace == DIRTY.length() && line.startsWith(DIRTY)) {
            entry.currentEditor = new Editor(entry);
        } else if (secondSpace != -1 || firstSpace != READ.length() || !line.startsWith(READ)) {
            throw new IOException("unexpected journal line: " + line);
        }
    }

    private void processJournal() throws IOException {
        DiskLruCache.deleteIfExists(this.journalFileTmp);
        Iterator<Entry> i = this.lruEntries.values().iterator();
        while (i.hasNext()) {
            Entry entry = i.next();
            if (entry.currentEditor == null) {
                this.size = entry.length;
                continue;
            }
            entry.currentEditor = null;
            DiskLruCache.deleteIfExists(entry.getCleanFile());
            DiskLruCache.deleteIfExists(entry.getDirtyFile());
            i.remove();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void rebuildJournal() throws IOException {
        if (this.journalWriter != null) {
            this.journalWriter.close();
        }
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(this.journalFileTmp), Util.US_ASCII));){
            writer.write(MAGIC);
            writer.write("\n");
            writer.write(VERSION_1);
            writer.write("\n");
            writer.write(Integer.toString(this.appVersion));
            writer.write("\n");
            writer.write("\n");
            for (Entry entry : this.lruEntries.values()) {
                if (entry.currentEditor != null) {
                    writer.write("DIRTY " + entry.key + '\n');
                    continue;
                }
                writer.write("CLEAN " + entry.key + entry.getLengths() + '\n');
            }
        }
        if (this.journalFile.exists()) {
            DiskLruCache.renameTo(this.journalFile, this.journalFileBackup, true);
        }
        DiskLruCache.renameTo(this.journalFileTmp, this.journalFile, false);
        this.journalFileBackup.delete();
        this.journalWriter = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(this.journalFile, true), Util.US_ASCII));
    }

    private static void deleteIfExists(File file) throws IOException {
        if (file.exists() && !file.delete()) {
            throw new IOException();
        }
    }

    private static void renameTo(File from, File to, boolean deleteDestination) throws IOException {
        if (deleteDestination) {
            DiskLruCache.deleteIfExists(to);
        }
        if (!from.renameTo(to)) {
            throw new IOException();
        }
    }

    public synchronized Snapshot get(String key) throws IOException {
        this.checkNotClosed();
        this.validateKey(key);
        Entry entry = this.lruEntries.get(key);
        if (entry == null) {
            return null;
        }
        if (!entry.readable) {
            return null;
        }
        File file = entry.getCleanFile();
        if (!file.exists() || !file.canRead()) {
            return null;
        }
        ++this.redundantOpCount;
        this.journalWriter.append("READ " + key + '\n');
        if (this.journalRebuildRequired()) {
            this.executorService.submit(this.cleanupCallable);
        }
        return new Snapshot(key, entry.sequenceNumber, file, entry.length);
    }

    public Editor edit(String key) throws IOException {
        return this.edit(key, -1L);
    }

    private synchronized Editor edit(String key, long expectedSequenceNumber) throws IOException {
        this.checkNotClosed();
        this.validateKey(key);
        Entry entry = this.lruEntries.get(key);
        if (expectedSequenceNumber != -1L && (entry == null || entry.sequenceNumber != expectedSequenceNumber)) {
            return null;
        }
        if (entry == null) {
            entry = new Entry(key);
            this.lruEntries.put(key, entry);
        } else if (entry.currentEditor != null) {
            return null;
        }
        Editor editor = new Editor(entry);
        entry.currentEditor = editor;
        this.journalWriter.write("DIRTY " + key + '\n');
        this.journalWriter.flush();
        return editor;
    }

    public File getDirectory() {
        return this.directory;
    }

    public synchronized long getMaxSize() {
        return this.maxSize;
    }

    public synchronized void setMaxSize(long maxSize) {
        this.maxSize = maxSize;
        this.executorService.submit(this.cleanupCallable);
    }

    public synchronized long size() {
        return this.size;
    }

    private synchronized void completeEdit(Editor editor, boolean success) throws IOException {
        Entry entry = editor.entry;
        if (entry.currentEditor != editor) {
            throw new IllegalStateException();
        }
        if (success && !entry.readable) {
            if (!editor.written) {
                editor.abort();
                throw new IllegalStateException("Newly created entry didn't create value");
            }
            if (!entry.getDirtyFile().exists()) {
                editor.abort();
                return;
            }
        }
        File dirty = entry.getDirtyFile();
        if (success) {
            if (dirty.exists()) {
                File clean = entry.getCleanFile();
                dirty.renameTo(clean);
                long oldLength = entry.length;
                long newLength = clean.length();
                entry.length = newLength;
                this.size = this.size - oldLength + newLength;
            }
        } else {
            DiskLruCache.deleteIfExists(dirty);
        }
        ++this.redundantOpCount;
        entry.currentEditor = null;
        if (entry.readable | success) {
            entry.readable = true;
            this.journalWriter.write("CLEAN " + entry.key + entry.getLengths() + '\n');
            if (success) {
                entry.sequenceNumber = this.nextSequenceNumber++;
            }
        } else {
            this.lruEntries.remove(entry.key);
            this.journalWriter.write("REMOVE " + entry.key + '\n');
        }
        this.journalWriter.flush();
        if (this.size > this.maxSize || this.journalRebuildRequired()) {
            this.executorService.submit(this.cleanupCallable);
        }
    }

    private boolean journalRebuildRequired() {
        int redundantOpCompactThreshold = 2000;
        return this.redundantOpCount >= 2000 && this.redundantOpCount >= this.lruEntries.size();
    }

    public synchronized boolean remove(String key) throws IOException {
        this.checkNotClosed();
        this.validateKey(key);
        Entry entry = this.lruEntries.get(key);
        if (entry == null || entry.currentEditor != null) {
            return false;
        }
        File file = entry.getCleanFile();
        if (file.exists() && !file.delete()) {
            throw new IOException("failed to delete " + file);
        }
        this.size -= entry.length;
        entry.length = 0L;
        ++this.redundantOpCount;
        this.journalWriter.append("REMOVE " + key + '\n');
        this.lruEntries.remove(key);
        if (this.journalRebuildRequired()) {
            this.executorService.submit(this.cleanupCallable);
        }
        return true;
    }

    public synchronized boolean isClosed() {
        return this.journalWriter == null;
    }

    private void checkNotClosed() {
        if (this.journalWriter == null) {
            throw new IllegalStateException("cache is closed");
        }
    }

    public synchronized void flush() throws IOException {
        this.checkNotClosed();
        this.trimToSize();
        this.journalWriter.flush();
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.journalWriter == null) {
            return;
        }
        for (Entry entry : new ArrayList<Entry>(this.lruEntries.values())) {
            if (entry.currentEditor == null) continue;
            entry.currentEditor.abort();
        }
        this.trimToSize();
        this.journalWriter.close();
        this.journalWriter = null;
    }

    private void trimToSize() throws IOException {
        while (this.size > this.maxSize) {
            Map.Entry<String, Entry> toEvict = this.lruEntries.entrySet().iterator().next();
            this.remove(toEvict.getKey());
        }
    }

    public void delete() throws IOException {
        this.close();
        Util.deleteContents(this.directory);
    }

    private void validateKey(String key) {
        Matcher matcher = LEGAL_KEY_PATTERN.matcher(key);
        if (!matcher.matches()) {
            throw new IllegalArgumentException("keys must match regex [\\.a-z0-9_-]{1,120}: \"" + key + "\"");
        }
    }

    private static String inputStreamToString(InputStream in) throws IOException {
        return Util.readFully(new InputStreamReader(in, Util.UTF_8));
    }

    static {
        LEGAL_KEY_PATTERN = Pattern.compile(STRING_KEY_PATTERN);
        NULL_OUTPUT_STREAM = new OutputStream(){

            @Override
            public void write(int b) throws IOException {
            }
        };
    }

    private final class Entry {
        private final String key;
        private long length;
        private boolean readable;
        private Editor currentEditor;
        private long sequenceNumber;

        private Entry(String key) {
            this.key = key;
        }

        public String getLengths() throws IOException {
            StringBuilder result = new StringBuilder();
            result.append(' ').append(DiskLruCache.this.size);
            return result.toString();
        }

        private void setLengths(String[] strings) throws IOException {
            try {
                for (int i = 0; i < strings.length; ++i) {
                    this.length = Long.parseLong(strings[i]);
                }
            }
            catch (NumberFormatException e) {
                throw this.invalidLengths(strings);
            }
        }

        private IOException invalidLengths(String[] strings) throws IOException {
            throw new IOException("unexpected journal line: " + Arrays.toString(strings));
        }

        private File getCleanFile() {
            return new File(DiskLruCache.this.directory, this.key);
        }

        public File getDirtyFile() {
            return new File(DiskLruCache.this.directory, this.key + ".tmp");
        }
    }

    public final class Editor {
        private final Entry entry;
        private boolean written;
        private boolean hasErrors;
        private boolean committed;

        private Editor(Entry entry) {
            this.entry = entry;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public RandomAccessOutput updateRandomAccessOutput() {
            DiskLruCache diskLruCache = DiskLruCache.this;
            synchronized (diskLruCache) {
                RandomAccessOutput output;
                if (this.entry.currentEditor != this) {
                    throw new IllegalStateException();
                }
                if (!this.entry.readable) {
                    this.written = true;
                } else {
                    this.copyCleanToDirty();
                }
                File dirtyFile = this.entry.getDirtyFile();
                try {
                    output = new RandomAccessOutput(dirtyFile);
                }
                catch (FileNotFoundException e) {
                    DiskLruCache.this.directory.mkdirs();
                    try {
                        output = new RandomAccessOutput(dirtyFile);
                    }
                    catch (FileNotFoundException e2) {
                        output = new RandomAccessOutput();
                    }
                }
                return new FaultHidingRandomAccessOutput(output);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Loose catch block
         */
        private void copyCleanToDirty() {
            File dirty = this.entry.getDirtyFile();
            File clean = this.entry.getCleanFile();
            FileInputStream is = null;
            FileOutputStream os = null;
            try {
                is = new FileInputStream(clean);
                os = new FileOutputStream(dirty);
                IOUtils.copy((InputStream)is, (OutputStream)os);
            }
            catch (IOException iOException) {
                IOUtils.closeQuietly((InputStream)is);
                IOUtils.closeQuietly(os);
                catch (Throwable throwable) {
                    IOUtils.closeQuietly(is);
                    IOUtils.closeQuietly(os);
                    throw throwable;
                }
            }
            IOUtils.closeQuietly((InputStream)is);
            IOUtils.closeQuietly((OutputStream)os);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public OutputStream newOutputStream() throws IOException {
            DiskLruCache diskLruCache = DiskLruCache.this;
            synchronized (diskLruCache) {
                FileOutputStream outputStream;
                if (this.entry.currentEditor != this) {
                    throw new IllegalStateException();
                }
                if (!this.entry.readable) {
                    this.written = true;
                }
                File dirtyFile = this.entry.getDirtyFile();
                try {
                    outputStream = new FileOutputStream(dirtyFile);
                }
                catch (FileNotFoundException e) {
                    DiskLruCache.this.directory.mkdirs();
                    try {
                        outputStream = new FileOutputStream(dirtyFile);
                    }
                    catch (FileNotFoundException e2) {
                        return NULL_OUTPUT_STREAM;
                    }
                }
                return new FaultHidingOutputStream(outputStream);
            }
        }

        public void commit() throws IOException {
            if (this.hasErrors) {
                DiskLruCache.this.completeEdit(this, false);
                DiskLruCache.this.remove(this.entry.key);
            } else {
                DiskLruCache.this.completeEdit(this, true);
            }
            this.committed = true;
        }

        public void abort() throws IOException {
            DiskLruCache.this.completeEdit(this, false);
        }

        public void abortUnlessCommitted() {
            if (!this.committed) {
                try {
                    this.abort();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
        }

        private class FaultHidingRandomAccessOutput
        extends RandomAccessOutput {
            private final RandomAccessOutput out;

            public FaultHidingRandomAccessOutput(RandomAccessOutput output) {
                this.out = output;
            }

            @Override
            public long length() throws IOException {
                try {
                    return this.out.length();
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                    return 0L;
                }
            }

            @Override
            public void seek(long pos) {
                try {
                    this.out.seek(pos);
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }

            @Override
            public void write(int oneByte) {
                try {
                    this.out.write(oneByte);
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }

            @Override
            public void write(byte[] buffer, int offset, int length) {
                try {
                    this.out.write(buffer, offset, length);
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }

            @Override
            public void close() {
                try {
                    this.out.close();
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }

            @Override
            public void flush() {
                try {
                    this.out.flush();
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }
        }

        private class FaultHidingOutputStream
        extends FilterOutputStream {
            private FaultHidingOutputStream(OutputStream out) {
                super(out);
            }

            @Override
            public void write(int oneByte) {
                try {
                    this.out.write(oneByte);
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }

            @Override
            public void write(byte[] buffer, int offset, int length) {
                try {
                    this.out.write(buffer, offset, length);
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }

            @Override
            public void close() {
                try {
                    this.out.close();
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }

            @Override
            public void flush() {
                try {
                    this.out.flush();
                }
                catch (IOException e) {
                    Editor.this.hasErrors = true;
                }
            }
        }
    }

    public final class Snapshot {
        private final String key;
        private final long sequenceNumber;
        private final File file;
        private final long length;

        private Snapshot(String key, long sequenceNumber, File file, long length) {
            this.key = key;
            this.sequenceNumber = sequenceNumber;
            this.file = file;
            this.length = length;
        }

        public Editor edit() throws IOException {
            return DiskLruCache.this.edit(this.key, this.sequenceNumber);
        }

        public File getFile() {
            return this.file;
        }

        public long getLength() {
            return this.length;
        }
    }
}

