package com.intsig.csopen.sdk;

import java.io.File;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.database.Cursor;
import android.net.Uri;
import android.text.TextUtils;

import com.intsig.csopen.util.Log;

final class CSOpenApiV1 implements CSOpenAPI{

	private static final String TAG = "CSOpenApi";
	
	private Context mContext;
	
	private String mAppKey;
	private String mUserId;
	
	private int mLastRequestCode;
	
	protected CSOpenApiV1(Context context, String appKey, String userId){
		mContext = context;
		mAppKey = appKey;
		mUserId = userId;
	}
	
	@Override
	public boolean isCamScannerAvailable() {
		float version = getOpenApiVersion();
		if(version >= 0){
			return true;
		}else{
			return false;
		}
	}
	
	public float getOpenApiVersion(){
		float version = -1.0f;
		Uri uri = Uri.withAppendedPath(Uri.parse(Const.SCHEME + Const.AUTHORITY), Const.VERSION_TABLE_NAME);
		Cursor cursor = mContext.getContentResolver().query(uri, null, null, null, null);
		if(cursor != null){
			if(cursor.moveToFirst()){
				version = cursor.getFloat(0);
			}
			cursor.close();
		}
		return version;
	}
	
	public boolean isCamScannerInstalled(){
		boolean res = false;
		try {
			PackageInfo pack = mContext.getPackageManager().
					getPackageInfo(Const.CS_PACK_NAME, PackageManager.GET_SIGNATURES);
			if(pack != null){
				res = true;
			}
		} catch (NameNotFoundException e) {
			Log.w(TAG, "CamScanner is not installed.");
		}
		return res;
	}

	@Override
	public boolean handleResult(int requestCode, int resultCode, Intent data, CSOpenApiHandler handler) {
		boolean res = false;
		if(mLastRequestCode == requestCode){
			// request code 匹配，表明可以处理该result
			res = true;
			if(data != null){	//如果该api需要用户登录；进入到CS是登录的操作的话，返回的data == null
				int responseCode = data.getIntExtra(Const.EXTRA_RESPONSE, -1);
				if(handler != null){
					if(resultCode == Activity.RESULT_OK){
						handler.onSuccess();
					}else if(resultCode == Activity.RESULT_FIRST_USER){
						handler.onError(responseCode);
					}else if(resultCode == Activity.RESULT_CANCELED){
						handler.onCancel();
					}
				}
			}
		}
		return res;
	}

	@Override
	public boolean scanImage(Activity activity, int requestCode, CSOpenAPIParam param) {
		if(param == null || !param.checkLegal()){
			Log.d(TAG, "param is illegal");
			return false;
		}
		File imgSrc = new File(param.mSourcePath);
		if(!imgSrc.exists() || imgSrc.length() < 1){
			Log.d(TAG, "file of imagePath not exist");
			return false;
		}
		if(!isCamScannerInstalled()){
			return false;
		}
		boolean res = false;
		Uri u = Uri.fromFile(imgSrc);
		
		// create intent to call CamScanner
		Intent i = new Intent(Const.ACTION_CAMSCANNER);
		i.putExtra(Intent.EXTRA_STREAM, u);		// put image resource
		i.putExtra(Const.EXTRA_OPEN_API_APIKEY, mAppKey);	// put app_key
		if(!TextUtils.isEmpty(mUserId)){
			i.putExtra(Const.EXTRA_OPEN_API_SUB_APPKEY, mUserId);
		}
		if(param.mMinRequiredApiVersion > 0){
			i.putExtra(Const.EXTRA_OPEN_API_VERSION, param.mMinRequiredApiVersion);	//put api_version
		}
		// 设置返回变量
		if(!TextUtils.isEmpty(param.mResultImgPath)){
			i.putExtra(Const.EXTRA_SCANNED_IMAGE, param.mResultImgPath);
		}
		if(!TextUtils.isEmpty(param.mResultPDFPath)){
			i.putExtra(Const.EXTRA_SCANNED_PDF, param.mResultPDFPath);
		}
		if(!TextUtils.isEmpty(param.mOriginalImagePath)){
			i.putExtra(Const.EXTRA_ORG_IMAGE, param.mOriginalImagePath);
		}

		try {
			mLastRequestCode = requestCode;
			activity.startActivityForResult(i, requestCode);
			res = true;
		} catch (ActivityNotFoundException e) {
			Log.e(TAG, "CamScanner is not Installed");
		}
		return res;
	}
	
	@Override
	public String getDownloadLink() {
		return "http://s.intsig.net/output/CamScanner_Lite_OpenAPI.apk";
	}
}
