package com.foresee.sdk;

import android.app.Application;

import com.foresee.sdk.common.Logging;
import com.foresee.sdk.common.configuration.IConfiguration;
import com.foresee.sdk.common.eventLogging.model.FeatureEvent;
import com.foresee.sdk.common.product.ForeSeeProduct;
import com.foresee.sdk.feedback.FeedbackContext;
import com.foresee.sdk.feedback.logging.LogTags;
import com.foresee.sdk.feedback.utils.FeatureLoggingHelper;
import com.foresee.sdk.internal.ForeSeeFeedbackFacade;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by alan.wang on 7/23/18.
 */

public class ForeSeeFeedback extends ForeSeeBase implements ForeSeeProduct {
    // region - Constants
    // endregion

    // region - Variables
    protected static ForeSeeFeedbackFacade instance;
    // endregion

    // region - Overrides
    @Override
    public boolean getIsStarted() {
        return instance !=  null;
    }

    @Override
    protected boolean prepareProduct(Application application, IConfiguration configuration) {
        boolean result = true;


        if (instance == null) {
            instance = new ForeSeeFeedbackFacade(application, configuration, FeedbackContext.getInstance());
            result = FeedbackContext.prepare(application, configuration);
        }
        return result;
    }

    @Override
    protected void completeProductInitialization(Application application, IConfiguration oldConfig,
                                                 IConfiguration newConfig) {

        FeedbackContext.getInstance().setConfiguration(newConfig);
    }

    @Override
    protected void resetProductState() {

    }

    @Override
    protected void initStubInstance() {

    }

    @Override
    protected void logFeatures(Application application, IConfiguration configuration) {


        FeatureLoggingHelper helper = new FeatureLoggingHelper(new FeatureEvent());
        helper.logFeaturesIfNecessary(application, configuration);
    }
    // endregion

    // region - Public instrumentation
    /**
     * Check if a feedback is enabled. The result will be returned via
     * {@see ForeSeeFeedbackListener#onFeedbackStatusRetrieved(String feedbackName, boolean enabled)}
     *
     * @param feedbackName a feedback name defined in the Configuration
     */
    public static void checkIfFeedbackEnabledForName(final String feedbackName) {
        submitRunnableOrAddToQueueIfSdkNotStarted(new Runnable() {
            public void run() {
                instance.checkIfFeedbackEnabledForName(feedbackName);
            }
        });
    }

    /**
     * Check if the default feedback is enabled. A default feedback is the first feedback defined
     * in the Configuration. The result will be returned via
     * {@see ForeSeeFeedbackListener#onFeedbackStatusRetrieved(String feedbackName, boolean enabled)}
     */
    public static void checkIfFeedbackEnabled() {
        submitRunnableOrAddToQueueIfSdkNotStarted(new Runnable() {
            public void run() {
                instance.checkIfFeedbackEnabled();
            }
        });
    }

    /**
     * Get all available feedback names defined in the Configuration.
     *
     * @return a list of available feedback names
     */
    public static List<String> getAvailableFeedbackNames() {
        if (!isForeSeeStarted()) {
            Logging.alwaysLog(Logging.LogLevel.WARN, LogTags.FEEDBACK,
                    "Unable to get available feedback names. Please call ForeSee.start() first");
            return null;
        } else {
            return instance.getAvailableFeedbackNames();
        }
    }

    /**
     * Set a feedback listener, which is an instance of {@link ForeSeeFeedbackListener}
     * @see ForeSeeFeedbackListener
     *
     * @param listener a feedback listener
     */
    public static void setFeedbackListener(final ForeSeeFeedbackListener listener) {
        submitRunnableOrAddToQueueIfSdkNotStarted(new Runnable() {
            public void run() {
                instance.setFeedbackListener(listener);
            }
        });
    }

    /**
     * Show the feedback for a given feedback name.
     *
     * @param feedbackName a feedback name
     */
    public static void showFeedbackForName(final String feedbackName) {
        submitRunnableOrAddToQueueIfSdkNotStarted(new Runnable() {
            public void run() {
                instance.showFeedbackForName(feedbackName);
            }
        });
    }

    /**
     * Show the default feedback. A default feedback is the first feedback defined
     * in the Configuration.
     */
    public static void showFeedback() {
        submitRunnableOrAddToQueueIfSdkNotStarted(new Runnable() {
            public void run() {
                instance.showFeedback();
            }
        });
    }
    // endregion


}
