package com.foresee.sdk;

import android.app.Activity;
import android.app.Application;
import android.os.Build;
import android.os.Looper;
import android.view.View;

import com.foresee.sdk.common.CoreContext;
import com.foresee.sdk.common.Logging;
import com.foresee.sdk.common.configuration.Configuration;
import com.foresee.sdk.common.configuration.ConfigurationLoader;
import com.foresee.sdk.common.configuration.ContactType;
import com.foresee.sdk.common.configuration.IConfiguration;
import com.foresee.sdk.common.constants.LogTags;
import com.foresee.sdk.common.constants.Values;
import com.foresee.sdk.common.eventLogging.EventLogger;
import com.foresee.sdk.common.eventLogging.model.Event;
import com.foresee.sdk.common.exception.ForeSeeUncaughtExceptionHandler;
import com.foresee.sdk.common.product.ForeSeeProduct;
import com.foresee.sdk.common.utils.Util;
import com.foresee.sdk.cxMeasure.tracker.TrackingContext;
import com.foresee.sdk.cxMeasure.tracker.listeners.BaseInviteListener;
import com.foresee.sdk.internal.ForeSeeFacade;
import com.foresee.sdk.internal.ForeSeeFacadeStub;
import com.google.gson.JsonSyntaxException;

import java.io.IOException;
import java.util.Locale;
import java.util.Map;

import static com.foresee.sdk.common.eventLogging.model.Event.*;
import static com.foresee.sdk.common.threading.SequentialExecution.getGlobalSequentialExecutor;

/**
 * This class contains the main API methods to be used by implementers when instrumenting their
 * application.
 */
public class ForeSee extends ForeSeeBase{
    protected static ForeSeeFacade instance;
    private static boolean skipPoolingCheckFlag;
    private static ForeSeeSDKConfigurationListener configurationLoaderListener;
    protected static boolean productsPrepared;

    // region - Public instrumentation - Initialization

    /**
     * Start the ForeSee SDK.
     * <p/>
     * This method will initialize the ForeSee SDK - by convention, the SDK will look for a
     * 'foresee_configuration.json' file in the 'assets' directory if the config wasn't set
     * beforehand.
     *
     * @param application the Application context
     */
    public static void start(final Application application) {
        start(application, null);
    }

    /**
     * Start the ForeSee SDK.
     * <p/>
     * This method will initialize the ForeSee SDK - by convention, the SDK will look for a
     * 'foresee_configuration.json' file in the 'assets' directory if the config wasn't set
     * beforehand.
     *
     * @param application the Application context
     * @param listener callback which notifies when the SDK started or failed to start
     */
    public static void start(final Application application, ForeSeeSDKConfigurationListener listener) {


        // Prepare products, but only if the products were not prepared already
        if (!prepareAllProductsIfNecessary(application)) {
            return;
        }

        configurationLoaderListener = listener;
        getGlobalSequentialExecutor().submit(new Runnable() {
            public void run() {
                Configuration configuration = ConfigurationLoader.getInstance().loadFromConfigFile(application);

                completeInitialization(application, configuration);

                if (configuration == null) {
                    EventLogger.logEvent(new Event(Event.EventType.SDKNotStarted).addProperty(Event.keyReason, Event.reasonConfigError));
                }
                else {
                    EventLogger.logEvent(new Event(Event.EventType.SDKStarted).addProperty(keyConfigurationMethod, valueDefault));
                }
            }
        });
    }

    /**
     * Starts the SDK using a custom configuration file
     * name.
     *
     * @param application the Application context
     * @param filename configuration filename
     */
    public static void startWithConfiguration(final Application application, final String filename) {
        startWithConfiguration(application, filename, null);
    }

    /**
     * Starts the SDK using a custom configuration file
     * name.
     *
     * @param application the Application context
     * @param filename configuration filename
     * @param listener callback which notifies when the SDK started or failed to start
     */
    public static void startWithConfiguration(final Application application, final String filename, ForeSeeSDKConfigurationListener listener) {
        // Prepare products, but only if the products were not prepared already
        if (!prepareAllProductsIfNecessary(application)) {
            return;
        }

        configurationLoaderListener = listener;
        getGlobalSequentialExecutor().submit(new Runnable() {
            public void run() {
                Configuration configuration = null;
                try {
                    configuration = ConfigurationLoader.getInstance().loadFromConfigFile
                            (application, filename);
                } catch (IOException e) {
                    Logging.alwaysLog(Logging.LogLevel.ERROR, LogTags.CONFIG, "Configuration file " + filename + " not found");
                } catch (JsonSyntaxException e) {
                    configuration = fallBackToLocalConfig(application, "Malformed JSON supplied");
                }
                configuration = validateConfiguration(configuration);

                completeInitialization(application, configuration);

                if (configuration == null) {
                    EventLogger.logEvent(new Event(Event.EventType.SDKNotStarted).addProperty(Event.keyReason, Event.reasonConfigError));
                }
                else {
                    EventLogger.logEvent(new Event(Event.EventType.SDKStarted).addProperty(keyConfigurationMethod, valueFile));
                }
            }
        });
    }

    /**
     * Starts the SDK using a custom configuration JSON string.
     *
     * @param application the Application context
     * @param configJSON JSON string for configuration
     */
    public static void startWithConfigurationJSON(final Application application, final String configJSON) {
        startWithConfigurationJSON(application, configJSON, null);
    }

    /**
     * Starts the SDK using a custom configuration JSON string.
     *
     * @param application the Application context
     * @param configJSON JSON string for configuration
     * @param listener callback which notifies when the SDK started or failed to start
     */
    public static void startWithConfigurationJSON(final Application application, final String configJSON, ForeSeeSDKConfigurationListener listener) {
        // Prepare products, but only if the products were not prepared already
        if (!prepareAllProductsIfNecessary(application)) {
            return;
        }

        configurationLoaderListener = listener;
        getGlobalSequentialExecutor().submit(new Runnable() {
            public void run() {
                Configuration configuration;
                try {
                    configuration = ConfigurationLoader.getInstance().loadFromJSON(configJSON);
                } catch (Exception e) {
                    configuration = fallBackToLocalConfig(application, "Malformed JSON supplied");
                }
                completeInitialization(application, configuration);

                if (configuration == null) {
                    EventLogger.logEvent(new Event(Event.EventType.SDKNotStarted).addProperty(Event.keyReason, Event.reasonConfigError));
                }
                else {
                    EventLogger.logEvent(new Event(Event.EventType.SDKStarted).addProperty(keyConfigurationMethod, valueJson));
                }
            }
        });
    }

    // endregion

    // region - Public instrumentation - Configuration updates

    /**
     * Updates the SDK with a custom configuration JSON string.
     *
     * @param application the Application context
     * @param configJSON JSON string for configuration
     * @return the new Configuration file that will be used
     */
    public static Configuration updateConfig(Application application, String configJSON) {
        if (!isForeSeeStarted()) {
            return null;
        } else {
            Configuration remoteConfiguration;
            try {
                remoteConfiguration = ConfigurationLoader.getInstance().loadFromJSON(configJSON);
                return updateConfig(application, remoteConfiguration);
            } catch (JsonSyntaxException e) {
                Logging.alwaysLog(Logging.LogLevel.ERROR, LogTags.CONFIG, "Malformed JSON " +
                        "supplied. Reverting to previous configuration.");
                return getSDKConfiguration();
            }
        }
    }

    /**
     * Updates the current persistenceSerializer to use a local configuration file. Locale changes will cause the
     * SDK to reload the local, region-specific configuration file.
     *
     * @param application the Application context
     * @return the new Configuration file that will be used
     */
    public static Configuration useLocalConfig(Application application) {
        if (!isForeSeeStarted()) return null;

        Configuration configuration = ConfigurationLoader.getInstance().loadFromConfigFile
                (application);
        if (configuration == null) {
            Logging.alwaysLog(Logging.LogLevel.ERROR, LogTags.CONFIG, "Unable to load local " +
                    "configuration file. Reverting to previous configuration.");
            return getSDKConfiguration();
        }
        return updateConfig(application, configuration);
    }

    /**
     * Listener to notify on completion or failure of the configuration process
     */
    public interface ForeSeeSDKConfigurationListener
    {
        /**
         * The SDK has been initialized successfully
         */
        void onSDKReady();

        /**
         * SDK initialization has failed due to a configuration error
         */
        void onFailedInitializingSDK();
    }

    // endregion
    // region - Internal initialization utility methods

    protected static boolean prepare(Application application) {
        // Sanity check the host app's Proguard settings
        if (!proguardSanityCheck()) {
            return false;
        }

        // This method contains only the bare necessities to get the SDK started
        // Only add code here that must be run synchronously as the app/SDK starts
        com.foresee.sdk.common.configuration.ConfigurationStub stopGapConfig = new com.foresee.sdk.common.configuration.ConfigurationStub();
        config = stopGapConfig;
        CoreContext.getInstance().setConfiguration(config);
        config.setSkipPooling(skipPoolingCheckFlag);
        config.setDebugLoggingEnabled(CoreContext.getInstance().getIsDebugLogEnabled());

        if (config.isPerfLoggingEnabled() == null) {
            config.setPerfLoggingEnabled(false);
        }

        boolean result = false;

        if (application == null)
            throw new IllegalArgumentException();

        // Store a reference of application to core context
        CoreContext.getInstance().setApplication(application);

        if (instance == null) {
            if (Build.VERSION.SDK_INT < Values.MINIMUM_SUPPORTED_ANDROID_VERSION) {
                Logging.alwaysLog(Logging.LogLevel.ERROR, LogTags.SDK_LIB, "Not initializing " +
                        "ForeSee SDK; Android API level " + Build.VERSION.SDK_INT + " is not supported");
                instance = new ForeSeeFacadeStub();
                for (ForeSeeProduct product : productList) {
                    ((ForeSeeBase)product).initStubInstance();
                }
            } else {
                result = true;
                for (ForeSeeProduct product : productList) {
                    result &= ((ForeSeeBase)product).prepareProduct(application, stopGapConfig);
                }

                // Init the event logger. Note: this must be called AFTER all products are prepared.
                EventLogger.initialize(application);

                // Init instance
                instance = new ForeSeeFacade(application);

                application.registerActivityLifecycleCallbacks(new ForeSeeLifecycleCallbacks());
            }
        }
        instance.prevLocale = application.getResources().getConfiguration().locale;

        if (result) {
            Logging.alwaysLog(Logging.LogLevel.INFO, LogTags.SDK_COMMON, "Initialized ForeSee SDK");
        } else {

        }
        return result;
    }

    /**
     * Prepare all ForeSee products (Core, CX Measure and Feedback, etc.) if they haven't prepared
     * already.
     * @param application - the application context
     * @return true if all products are successfully prepared, otherwise false
     */
    static boolean prepareAllProductsIfNecessary(Application application) {
        if (!productsPrepared) {
            // TODO: we should change this and let the host apps to add products manually -
            // this will be handled in MOBILSDK-1626.
            // Add products
            addProducts();
            productsPrepared = prepare(application);
        }
        return productsPrepared;
    }

    static void completeInitialization(Application application, Configuration configuration) {
        // All start methods call this one last; put any asynchronous start code here
        Thread.currentThread().setName("ForeSee sequential tasks");
        updateConfig(application, configuration);

        Logging.alwaysLog(Logging.LogLevel.INFO, LogTags.SDK_COMMON, String.format(Locale.CANADA, "Loaded configuration and state for ForeSee SDK v%s", instance.getVersion()));
        if (configurationLoaderListener != null) {
            new android.os.Handler(Looper.getMainLooper()).post(new Runnable() {
                @Override
                public void run() {
                    if (isForeSeeStarted()) {
                        configurationLoaderListener.onSDKReady();
                    } else {
                        configurationLoaderListener.onFailedInitializingSDK();
                    }
                }
            });
        }

        if (isForeSeeStarted()) {
            // Setup a ForeSee default uncaught exception handler
            Thread.setDefaultUncaughtExceptionHandler(ForeSeeUncaughtExceptionHandler.getInstance());

            // Submit all unsubmitted tasks
            submitUnsubmittedRunnableQueue();
        } else {
            // Clear all unsubmitted tasks since SDK failed to initialize
            clearUnsubmittedRunnableQueue();
        }
    }

    @Override
    protected boolean prepareProduct(Application application, IConfiguration configuration) {
        // no-op
        Logging.alwaysLog(Logging.LogLevel.INFO, LogTags.SDK_COMMON,
                "ForeSee.prepareProduct - empty method. We shouldn't be here.");
        return false;
    }

    @Override
    protected void completeProductInitialization(Application application,
                                                 IConfiguration oldConfig,
                                                 IConfiguration newConfig) {
        // no-op
        Logging.alwaysLog(Logging.LogLevel.INFO, LogTags.SDK_COMMON,
                "ForeSee.completeProductInitialization - empty method. We shouldn't be here.");
    }

    @Override
    protected void resetProductState() {
        // no-op
        Logging.alwaysLog(Logging.LogLevel.INFO, LogTags.SDK_COMMON,
                "ForeSee.resetProductState - empty method. We shouldn't be here.");
    }

    @Override
    protected void initStubInstance() {
        // no-op
        Logging.alwaysLog(Logging.LogLevel.INFO, LogTags.SDK_COMMON,
                "ForeSee.initStubInstance - empty method. We shouldn't be here.");
    }

    @Override
    protected void logFeatures(Application application, IConfiguration configuration) {
        // no-op
        Logging.alwaysLog(Logging.LogLevel.INFO, LogTags.SDK_COMMON,
                "ForeSee.initStubInstance - empty method. We shouldn't be here.");
    }
    // endregion
    // region - Internal configuration utility methods

    private static Configuration validateConfiguration(Configuration configuration) {
        if (configuration == null) {
            Logging.alwaysLog(Logging.LogLevel.ERROR, LogTags.CONFIG, "Unable to load local configuration file. Starting SDK with no measures.");
            try {
                configuration = ConfigurationLoader.getInstance().loadWithoutMeasures();
            } catch (JsonSyntaxException e) {
                Logging.alwaysLog(Logging.LogLevel.ERROR, LogTags.CONFIG, "Unable to load configuration file with no measures.");
            }
        }
        return configuration;
    }

    private static Configuration updateConfig(Application application, Configuration newConfiguration) {
        if (newConfiguration != null) {


            if (config instanceof com.foresee.sdk.common.configuration.ConfigurationStub) {
                newConfiguration.setSkipPooling(config.shouldSkipPoolingCheck());
                newConfiguration.setDebugLoggingEnabled(config.isDebugLoggingEnabled());
                newConfiguration.setPerfLoggingEnabled(config.isPerfLoggingEnabled());
                TrackingContext.start(application);
            }
            else {
                if (config.shouldSkipPoolingCheck() != null
                        && newConfiguration.shouldSkipPoolingCheck() == null) {
                    newConfiguration.setSkipPooling(config.shouldSkipPoolingCheck());
                }

                if (config.isDebugLoggingEnabled() != null
                        && newConfiguration.isDebugLoggingEnabled() == null) {
                    newConfiguration.setDebugLoggingEnabled(config.isDebugLoggingEnabled());
                }

                if (config.isPerfLoggingEnabled() != null
                        && newConfiguration.isPerfLoggingEnabled() == null) {
                    newConfiguration.setPerfLoggingEnabled(config.isPerfLoggingEnabled());
                }
            }

            IConfiguration oldConfiguration = config;
            config = newConfiguration;
            CoreContext.getInstance().setConfiguration(config);
            config.loadPersistedCPPs();

            // Iterate products
            for (ForeSeeProduct product : productList) {
                ((ForeSeeBase)product).completeProductInitialization(application, oldConfiguration, newConfiguration);
                ((ForeSeeBase)product).logFeatures(application, newConfiguration);
            }
        }

        return getSDKConfiguration();
    }

    private static Configuration fallBackToLocalConfig(Application application, String reason) {
        Logging.alwaysLog(Logging.LogLevel.ERROR, LogTags.SDK_LIB, reason +
                "; Attempting to fall back to local configuration file.");
        return validateConfiguration(ConfigurationLoader.getInstance()
                .loadFromConfigFile(application));
    }

    private static boolean proguardSanityCheck() {
        boolean result = true;
        final String[] dontObfuscateClasses = {
                "com.foresee.sdk.ForeSee",
                "com.foresee.sdk.ForeSeeCxMeasure",
                "com.foresee.sdk.common.configuration.Configuration"
        };
        for (String entry : dontObfuscateClasses) {
            if (!Util.checkClassExistence(entry)) {
                Logging.alwaysLog(Logging.LogLevel.ERROR, LogTags.SDK_LIB, "It appears " +
                        "your Proguard settings for the ForeSee SDK are not set correctly. " +
                        "To fix this, please follow the instructions described in " +
                        "https://developer.foresee.com/docs/proguard.");
                result = false;
                break;
            }
        }
        return result;
    }

    // endregion
    // region - Public instrumentation - SDK notification - cxMeasure

    /**
     * Increment the significant event count for a given key.
     *
     * @param key the key to increment the significant event count for
     */
    public static void incrementSignificantEventCountWithKey(final String key) {
        ForeSeeCxMeasure.incrementSignificantEventCountWithKey(key);
    }

    /**
     * Manually increment the number of pages counted by the ForeSee SDK. This can be useful where the user would consider
     * a new page to have been shown, but the activity has not been changed, e.g. when using full screen fragments
     */
    public static void incrementPageViews() {
        ForeSeeCxMeasure.incrementPageViews();
    }

    /**
     * @deprecated Add contact details for use in the CONTACT notification method
     * @param contactDetails An email address or mobile number to use. Format should be name@domain.com, 555-123-1234, 5551231234, etc.
     */
    @Deprecated
    public static void setContactDetails(String contactDetails) {
        ForeSeeCxMeasure.setContactDetails(contactDetails);
    }

    /**
     * Add contact details for a particular type for use in the CONTACT notification method
     * @param contactType A particular contact type
     * @param contactDetails An email address or mobile number to use. Format should be name@domain.com, 555-123-1234, 5551231234, etc.
     */
    public static void setContactDetails(ContactType contactType, String contactDetails) {
        ForeSeeCxMeasure.setContactDetails(contactType, contactDetails);
    }

    /**
     * Set a preferred contact type
     * @param contactType A particular contact type
     */
    public static void setPreferredContactType(final ContactType contactType) {
        ForeSeeCxMeasure.setPreferredContactType(contactType);
    }

    /**
     * Adds a CPP key/value pair. CPPs are transmitted along with surveys upon submission.
     *
     * @param key   the key to store the CPP under
     * @param value the data to store in the CPP
     */
    public static void addCPPValue(final String key, final String value) {
        submitRunnableOrAddToQueueIfSdkNotStarted(new Runnable() {
            public void run() {
                config.addCpp(key, value, true);
            }
        });
    }

    /**
     * Removes a CPP value.
     *
     * @param key the key of the CPP to remove
     */
    public static void removeCPPValue(final String key) {
        submitRunnableOrAddToQueueIfSdkNotStarted(new Runnable() {
            public void run() {
                config.removeCpp(key);
            }
        });
    }

    /**
     * Adds a CPP key/value pair. The method takes a string key and an array, overwrites
     * the key with the contents of the array, comma separated. CPPs are transmitted along
     * with surveys upon submission.
     *
     * @param key   the key to store the CPP under
     * @param values the value array to store in the CPP
     */
    public static void setCPPValueFromArray(final String key, final String[] values) {
        submitRunnableOrAddToQueueIfSdkNotStarted(new Runnable() {
            public void run() {
                config.setCPPValueFromArray(key, values, true);

                EventLogger.logEvent(new Event(Event.EventType.CustomCPPSetFromArray));
            }
        });
    }

    /**
     * Appends a value to a comma separated list stored against a custom CPP key. CPPs are
     * transmitted along with surveys upon submission.
     *
     * @param key   the key to store the CPP under
     * @param value the value to append to the CPP
     */
    public static void appendCPPValue(final String key, final String value) {
        submitRunnableOrAddToQueueIfSdkNotStarted(new Runnable() {
            public void run() {
                config.appendCPPValue(key, value, true);

                EventLogger.logEvent(new Event(Event.EventType.CustomCPPAppended));
            }
        });
    }
    // endregion
    // region - Public Instrumentation - SDK notification - Custom invites and invite listeners

    /**
     * Set the invite listener
     * @param inviteListener - Choose the listener based on your notification type
     *                       Options are: CustomExitSurveyInviteListener | CustomInSessionInviteListener | CustomContactInviteListener OR DefaultInviteListener if you are not customizing the invitation UI
     */
    public static void setInviteListener(BaseInviteListener inviteListener) {
        ForeSeeCxMeasure.setInviteListener(inviteListener);
    }

    /**
     * Tells the SDK that a custom invite was accepted.
     * You should call this method whenever a user accepts a custom invite that you've presented.
     */
    public static void customInviteAccepted() {
        ForeSeeCxMeasure.customInviteAccepted();
    }

    /**
     * Tells the SDK that a custom invite was declined.
     * You should call this method whenever a user declines a custom invite that you've presented.
     */
    public static void customInviteDeclined() {
        ForeSeeCxMeasure.customInviteDeclined();
    }

    // endregion
    // region - Public instrumentation - SDK inquiry - cxMeasure

    /**
     * Gets the version of the SDK
     *
     * @return the version of the SDK
     */
    public static String getVersion() {
        return SDKConfig.FORESEE_SDK_VERSION;
    }


    /**
     * @deprecated Check the preferred contact details stored in the SDK
     * @return The stored preferred contact details
     */
    @Deprecated
    public static String getContactDetails() {
        Logging.alwaysLog(Logging.LogLevel.WARN, LogTags.CONFIG,
                "getContactDetails() is now deprecated. Please call getContactDetails(contactType) " +
                        "or getAllContactDetails() instead.");
        return ForeSeeCxMeasure.getContactDetails();
    }

    /**
     * Check the contact details for the given contact type stored in the SDK
     * @param contactType A given contact type
     * @return The stored contact details for the given contact type
     */
    public static String getContactDetails(ContactType contactType) {
        return ForeSeeCxMeasure.getContactDetails(contactType);
    }

    /**
     * Check all contact details
     * @return The stored contact details
     */
    public static Map<ContactType, String> getAllContactDetails() {
        return ForeSeeCxMeasure.getAllContactDetails();
    }

    /**
     * Get the preferred contact type
     * @return The stored preferred contact type
     */
    public static ContactType getPreferredContactType() {
        return ForeSeeCxMeasure.getPreferredContactType();
    }

    /**
     * Gets the CPP value associated to a given key
     * @param key   the key of the CPP
     * @return The data stored in the CPP, or null if SDK is not started or the CPP for the key does not exist
     */
    public static String getCPPValue(final String key) {
        return ForeSeeCxMeasure.getCPPValue(key);
    }

    /**
     * Gets all CPP key/value pairs
     * @return A map that contains all CPP key/value pairs, or null if SDK is not started
     */
    public static Map<String, String> getAllCPPs() {
        return ForeSeeCxMeasure.getAllCPPs();
    }

    // endregion
    // region - Public instrumentation - SDK Control - cxMeasure

    /**
     * Check to see if the user is eligible for a survey. If the user meets CX Measure triggering criteria *and*
     * are in the sampling pool, the invitation will be presented. Implementers must explicitly
     * check
     * for eligibility (the SDK will not do this automatically).
     */
    public static void checkIfEligibleForSurvey() {
        ForeSeeCxMeasure.checkIfEligibleForSurvey();
    }

    /**
     * Programmatically present the invitation for a given survey ID (sid)
     *
     * @param surveyId the survey id to present the invitation for.
     */
    public static void showInviteForSurveyID(final String surveyId) {
        ForeSeeCxMeasure.showInviteForSurveyID(surveyId);
    }

    /**
     * Programmatically present the survey for a given survey ID (sid)
     *
     * @param surveyId the survey id to present the survey for.
     */
    public static void showSurveyForSurveyID(final String surveyId) {
        ForeSeeCxMeasure.showSurveyForSurveyID(surveyId);
    }

    // endregion
    // region - Public instrumentation - Debug

    /**
     * Reset the state of the SDK.
     */
    public static void resetState() {
        if (!isForeSeeStarted()) {
            return;
        }

        EventLogger.logEvent(new Event(Event.EventType.TrackerReset));

        for (ForeSeeProduct product : productList) {
            ((ForeSeeBase)product).resetProductState();
        }


    }

    /**
     * Allows implementers to configure the SDK to skip the pooling check. This is often useful
     * during development but *NOT* recommended for production use. By default, the pooling check is
     * always enabled.
     *
     * @param skipPoolingCheck if set to true, skips the pooling check, otherwise the pooling check
     *                         is performed.
     */
    public static void setSkipPoolingCheck(boolean skipPoolingCheck) {
        if (skipPoolingCheck) EventLogger.logEvent(new Event(Event.EventType.SkipPoolingCheck));
        skipPoolingCheckFlag = skipPoolingCheck;
        if (config != null) {
            config.setSkipPooling(skipPoolingCheckFlag);
        }
    }

    /**
     * Check if the pooling check is disabled
     *
     * @return TRUE if the pooling check is disabled
     */
    public boolean shouldSkipPoolingCheck() {
        return skipPoolingCheckFlag;
    }

    /**
     * Allows implementers to configure the SDK to produce more verbose logging output. If enabled,
     * output from the SDK will be output to LogCat.
     *
     * @param debugLoggingEnabled if set to true, output from the SDK will be output to LogCat
     */
    public static void setDebugLogEnabled(boolean debugLoggingEnabled) {
        CoreContext.getInstance().setIsDebugLogEnabled(debugLoggingEnabled);
        if (config != null) {
            config.setDebugLoggingEnabled(debugLoggingEnabled);
        }
    }

    /**
     * Check whether debug logging is enabled
     *
     * @return TRUE if debug logging is enabled
     */
    public static boolean isDebugLogEnabled() {
        return CoreContext.getInstance().getIsDebugLogEnabled();
    }

    // endregion
    // region - Deprecated methods

    /**
     * Register a fragment view to ensure that its contents are processed correctly
     *
     * @deprecated This method was deprecated in version 4.1
     * @param fragmentView the view of the fragment to be registered
     */
    public static void registerFragmentView(final View fragmentView) {
    }

    /**
     * Lifecycle instrumentation for notifying the SDK that an Activity has started.
     *
     * @deprecated This method was deprecated in version 3.4.3
     * @param activity the Activity that started
     */
    public static void activityStarted(Activity activity) {
        showLifecycleWarning();
    }

    /**
     * Lifecycle instrumentation for notifying the SDK that an Activity has been resumed.
     *
     * @deprecated This method was deprecated in version 3.4.3
     * @param activity the Activity that has resumed.
     */
    public static void activityResumed(Activity activity) {
        showLifecycleWarning();
    }

    /**
     * Lifecycle instrumentation for notifying the SDK that an Activity has been paused.
     *
     * @deprecated This method was deprecated in version 3.4.3
     * @param activity the Activity that was paused.
     */
    public static void activityPaused(Activity activity) {
        showLifecycleWarning();
    }

    private static void showLifecycleWarning() {
        Logging.alwaysLog(Logging.LogLevel.WARN, LogTags.SDK_LIB, "Lifecycle hooks are no longer required. Please " +
                "remove all references to activityStarted, activityResumed and activityPaused");
    }

    // endregion

    // region - Remote config methods (i.e., Adobe extension related)
    /**
     * Start the ForeSee SDK with a late enable flag. The SDK will start but will only be enabled
     * later after the late enable flag is set.
     * <p/>
     * This method will initialize the ForeSee SDK - by convention, the SDK will look for a
     * 'foresee_configuration.json' file in the 'assets' directory if the config wasn't set
     * beforehand.
     *
     * @param application the Application context
     * @param listener callback which notifies when the SDK started or failed to start
     */
    static void startWithLateEnableFlag(final Application application, ForeSeeSDKConfigurationListener listener) {


        // Disable the SDK for now. And late enable it when we can after confirming the remote configuration
        ForeSee.disable();

        start(application, listener);
    }
    // endregion
}
