/*
 * Decompiled with CFR 0.152.
 */
package com.foilen.smalltools.tools;

import com.foilen.smalltools.exception.SmallToolsException;
import com.foilen.smalltools.tools.CharsetTools;
import com.foilen.smalltools.tools.DateTools;
import com.foilen.smalltools.tools.FileTools;
import com.google.common.base.Joiner;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Stack;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DirectoryTools {
    private static final Logger logger = LoggerFactory.getLogger(DirectoryTools.class);

    public static String cleanupDots(String path) {
        Stack<String> stack = new Stack<String>();
        for (String part : path.split("/")) {
            if (".".equals(part)) continue;
            if ("..".equals(part)) {
                if (stack.isEmpty()) continue;
                stack.pop();
                continue;
            }
            stack.push(part);
        }
        Joiner joiner = Joiner.on((String)"/");
        Object result = joiner.join(stack);
        if (path.startsWith("/") && !((String)result).startsWith("/")) {
            result = "/" + (String)result;
        }
        return result;
    }

    public static boolean createPath(File directory) {
        if (!directory.exists()) {
            return directory.mkdirs();
        }
        return true;
    }

    public static boolean createPath(String directoryPath) {
        return DirectoryTools.createPath(new File(directoryPath));
    }

    public static boolean createPath(String directoryPath, String owner, String group, String permissions) {
        logger.debug("createPath {} ", (Object)directoryPath);
        if (!DirectoryTools.createPath(directoryPath)) {
            return false;
        }
        FileTools.changeOwnerAndGroup(directoryPath, false, owner, group);
        FileTools.changePermissions(directoryPath, false, permissions);
        return true;
    }

    public static boolean createPath(String[] directoryPathParts) {
        return DirectoryTools.createPath(new File(FileTools.concatPath(directoryPathParts)));
    }

    public static boolean createPath(String[] directoryPathParts, String owner, String group, String permissions) {
        return DirectoryTools.createPath(FileTools.concatPath(directoryPathParts), owner, group, permissions);
    }

    public static boolean createPathAndCopyOwnerAndGroupFromParent(File directory) {
        if (!directory.exists()) {
            File parentDirectory = directory.getParentFile();
            if (!parentDirectory.exists() && !DirectoryTools.createPathAndCopyOwnerAndGroupFromParent(parentDirectory)) {
                return false;
            }
            if (!directory.mkdir()) {
                logger.error("Could not create directory {}", (Object)directory.getAbsolutePath());
                return false;
            }
            FileTools.copyOwnerAndGroupFromParentDir(directory, false);
        }
        return true;
    }

    public static boolean createPathAndCopyOwnerAndGroupFromParent(String directoryPath) {
        return DirectoryTools.createPathAndCopyOwnerAndGroupFromParent(new File(directoryPath));
    }

    public static boolean createPathAndCopyOwnerAndGroupFromParent(String[] directoryPathParts) {
        return DirectoryTools.createPathAndCopyOwnerAndGroupFromParent(new File(FileTools.concatPath(directoryPathParts)));
    }

    public static boolean createPathToFile(String filePath) {
        int windowPos;
        int unixPos = filePath.lastIndexOf("/");
        int endOfDirectoryPos = Math.max(unixPos, windowPos = filePath.lastIndexOf("\\"));
        if (endOfDirectoryPos == -1) {
            return true;
        }
        String directoryPath = filePath.substring(0, endOfDirectoryPos);
        return DirectoryTools.createPath(new File(directoryPath));
    }

    public static int deleteEmptySubFolders(File rootFolder) {
        AtomicInteger count = new AtomicInteger();
        DirectoryTools.visitFilesAndFoldersRecursively(rootFolder, (File item) -> {
            if (item.isDirectory()) {
                logger.debug("Checking if directory {} is empty", (Object)item.getPath());
                if (DirectoryTools.safeListFiles(item).length == 0) {
                    logger.info("Deleting directory {} because it is empty", (Object)item.getPath());
                    count.incrementAndGet();
                    if (!item.delete()) {
                        logger.error("Could not delete folder {}", (Object)item.getPath());
                    }
                }
            }
        });
        return count.get();
    }

    public static int deleteEmptySubFolders(String rootFolder) {
        return DirectoryTools.deleteEmptySubFolders(new File(rootFolder));
    }

    public static void deleteFolder(File folder) {
        logger.info("Delete folder {}", (Object)folder.getAbsolutePath());
        if (!folder.exists()) {
            return;
        }
        if (folder.isDirectory()) {
            String rootDir = folder.getAbsolutePath() + File.separator;
            for (File toDelete : DirectoryTools.safeListFiles(folder)) {
                DirectoryTools.deleteSub(rootDir, toDelete);
            }
        }
        if (!folder.delete()) {
            throw new SmallToolsException("Could not delete " + folder.getAbsolutePath());
        }
    }

    public static void deleteFolder(String folderPath) {
        DirectoryTools.deleteFolder(new File(folderPath));
    }

    public static int deleteOlderFilesInDirectory(File rootFolder, Date beforeDate) {
        AtomicInteger count = new AtomicInteger();
        long expiredBefore = beforeDate.getTime();
        DirectoryTools.visitFilesAndFoldersRecursively(rootFolder, (File item) -> {
            if (item.isFile()) {
                logger.debug("Checking if file {} is too old", (Object)item.getPath());
                if (item.lastModified() < expiredBefore) {
                    logger.info("Deleting file {} . Last modified time: {}", (Object)item.getPath(), (Object)DateTools.formatFull(new Date(item.lastModified())));
                    count.incrementAndGet();
                    if (!item.delete()) {
                        logger.error("Could not delete file {}", (Object)item.getPath());
                    }
                }
            }
        });
        return count.get();
    }

    public static int deleteOlderFilesInDirectory(String rootFolder, Date beforeDate) {
        return DirectoryTools.deleteOlderFilesInDirectory(new File(rootFolder), beforeDate);
    }

    private static void deleteSub(String rootDir, File folder) {
        String toDelete = folder.getAbsolutePath();
        if (!toDelete.startsWith(rootDir)) {
            throw new SmallToolsException("Trying to delete recursively the folder [" + rootDir + "] we got to delete [" + toDelete + "] which is not a direct child");
        }
        if (!Files.isSymbolicLink(folder.toPath()) && folder.isDirectory()) {
            for (File toDeleteSub : DirectoryTools.safeListFiles(folder)) {
                DirectoryTools.deleteSub(rootDir, toDeleteSub);
            }
        }
        if (!folder.delete()) {
            throw new SmallToolsException("Could not delete " + toDelete);
        }
    }

    public static List<String> listFilesAndFoldersRecursively(File directory, boolean absolute) {
        if (!directory.isDirectory()) {
            throw new SmallToolsException(directory.getAbsolutePath() + " is not a directory");
        }
        int relativeStartPos = directory.getAbsolutePath().length() + 1;
        List<String> results = DirectoryTools.listFilesAndFoldersRecursively(directory, absolute, relativeStartPos);
        Collections.sort(results);
        return results;
    }

    private static List<String> listFilesAndFoldersRecursively(File directory, boolean absolute, int relativeStartPos) {
        ArrayList<String> results = new ArrayList<String>();
        for (File file : DirectoryTools.safeListFiles(directory)) {
            if (file.isFile()) {
                if (absolute) {
                    results.add(file.getAbsolutePath());
                } else {
                    results.add(file.getAbsolutePath().substring(relativeStartPos));
                }
            }
            if (!file.isDirectory()) continue;
            if (absolute) {
                results.add(file.getAbsolutePath() + "/");
            } else {
                results.add(file.getAbsolutePath().substring(relativeStartPos) + "/");
            }
            results.addAll(DirectoryTools.listFilesAndFoldersRecursively(file, absolute, relativeStartPos));
        }
        return results;
    }

    public static List<String> listFilesAndFoldersRecursively(String path, boolean absolute) {
        return DirectoryTools.listFilesAndFoldersRecursively(new File(path), absolute);
    }

    public static List<String> listFilesStartingWith(String path, String startText) {
        File directory = new File(path);
        if (!directory.isDirectory()) {
            throw new SmallToolsException(path + " is not a directory");
        }
        byte[] startBytes = startText.getBytes(CharsetTools.UTF_8);
        ArrayList<String> result = new ArrayList<String>();
        for (File file : DirectoryTools.safeListFiles(directory)) {
            if (!file.isFile()) continue;
            try (FileInputStream inputStream = new FileInputStream(file);){
                byte[] buffer = new byte[startBytes.length];
                ((InputStream)inputStream).read(buffer);
                if (!Arrays.equals(startBytes, buffer)) continue;
                result.add(file.getName());
            }
            catch (Exception e) {
                throw new SmallToolsException("Could not read file " + file.getAbsolutePath(), e);
            }
        }
        Collections.sort(result);
        return result;
    }

    public static List<String> listOnlyFileNames(String directory) {
        return Arrays.asList(DirectoryTools.safeListFiles(new File(directory))).stream().filter(File::isFile).map(File::getName).sorted().collect(Collectors.toList());
    }

    public static String pathTrailingSlash(String path) {
        if (!((String)path).isEmpty() && ((String)path).charAt(((String)path).length() - 1) != '/') {
            path = (String)path + "/";
        }
        return path;
    }

    public static File[] safeListFiles(File directory) {
        File[] files = directory.listFiles();
        return files == null ? new File[]{} : files;
    }

    public static void visitFilesAndFoldersRecursively(File directory, Consumer<File> fileAction) {
        if (!directory.isDirectory()) {
            throw new SmallToolsException(directory.getAbsolutePath() + " is not a directory");
        }
        for (File file : DirectoryTools.safeListFiles(directory)) {
            if (file.isFile()) {
                fileAction.accept(file);
            }
            if (!file.isDirectory()) continue;
            DirectoryTools.visitFilesAndFoldersRecursively(file, fileAction);
            fileAction.accept(file);
        }
    }

    public static void visitFilesAndFoldersRecursively(String directory, Consumer<File> fileAction) {
        DirectoryTools.visitFilesAndFoldersRecursively(new File(directory), fileAction);
    }

    private DirectoryTools() {
    }
}

