package com.flybits.context

import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.os.Handler
import android.os.Looper
import androidx.work.*
import com.flybits.commons.library.api.FlyAway
import com.flybits.commons.library.api.results.BasicResult
import com.flybits.commons.library.api.results.ObjectResult
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.exceptions.*
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.logging.Logger.appendTag
import com.flybits.commons.library.logging.Logger.exception
import com.flybits.commons.library.models.User
import com.flybits.commons.library.models.internal.Result
import com.flybits.context.db.ContextDatabase.Companion.getDatabase
import com.flybits.context.models.ContextData
import com.flybits.context.models.ContextPriority
import com.flybits.context.plugins.FlybitsContextPlugin
import com.flybits.context.services.FlybitsContextPluginService
import com.flybits.context.services.FlybitsContextPluginsWorker
import com.flybits.context.utils.ContextUtilities
import com.flybits.context.workers.ContextUploadingWorker
import com.flybits.internal.db.CommonsDatabase
import com.google.android.gms.gcm.GcmNetworkManager
import com.google.android.gms.gcm.OneoffTask
import com.google.android.gms.gcm.PeriodicTask
import com.google.android.gms.gcm.Task
import kotlinx.coroutines.*
import java.util.*
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

private const val TAG_CM = "CtxManager"
private const val API_CONTEXT_DATA = ContextScope.ROOT + "/ctxdata"
private const val KEY_PRIORITY = "priority"

/**
 * Responsible for managing the Context Plugins and managing services initiated by [ContextScope]
 *
 */
object ContextManager {

    private val job = SupervisorJob()
    private val coroutineScope = CoroutineScope(Dispatchers.Default + job)

    private fun exceptionHandlerBuilder(text: String = "Error executing the CoroutineScope") =
        CoroutineExceptionHandler { _, _ ->
            appendTag(TAG_CM).e(text)
        }

    /**
     * Get Context Plug-in data that is stored within the Flybits Context DB. It's important to note
     * that `pluginID` and `subclass` must match, otherwise an empty [ContextData]
     * object will be returned or null if there was an error.
     *
     * @param context  The context of the application.
     * @param pluginID The unique identifier that represents a specific [ContextPlugin].
     * @param subclass The class that should be instantiated and context data should be parsed into.
     * @param callback The `ObjectResultCallback` that indicates whether or not the data for
     * `pluginID`, which could null.
     * @param <T>      The generic that represents the a [ContextData] class which can parse
     * the Context values.
     * @return The [ContextData] that the [ContextPlugin] data is parsed into.
     * @throws InvalidParameterException when params are passed as null
     * @throws InstantiationException when subclass specified in parameter should have an empty constructor.
     * @throws FlybitsException when other exception encountered.
    </T> */
    @JvmStatic
    @Throws(InvalidParameterException::class)
    fun <T : ContextData> getData(
        context: Context,
        pluginID: String,
        subclass: Class<T>,
        callback: ObjectResultCallback<T>?
    ): ObjectResult<T>? {
        if (context == null) {
            throw InvalidParameterException("context", "The Context parameter must not be null")
        }
        if (pluginID == null) {
            throw InvalidParameterException("pluginId", "The pluginId parameter must not be null")
        }
        if (subclass == null) {
            throw InvalidParameterException("subclass", "The Class parameter must not be null")
        }
        try {
            subclass.newInstance() as T
        } catch (e: Exception) {
            if (e is InstantiationException) {
                throw InstantiationException(
                    "The .Class file that will be instantiated must contain a constructor with no parameter for this method to be successfully processed"
                )
            } else {
                throw FlybitsException(e.message)
            }
        }
        return getDataPrivate(context, pluginID, subclass, callback)
    }

    /**
     * Retrieves all the Context information from the local DB and sends it to the server if it is
     * determined that new Context data has been received.
     *
     * @param context  The state of the application.
     * @param callback The `BasicResultCallback` that indicates whether or not the request was
     * successful or not. If not, a reason will be provided.
     * @return The `BasicResult` which contains information about the request being made.
     * @throws NetworkResponseException when server call to flush the context data fails.
     */
    @JvmStatic
    fun flushContextData(
        context: Context,
        callback: BasicResultCallback?
    ): BasicResult {
        val handler = Handler(Looper.getMainLooper())
        val executorService =
            Executors.newSingleThreadExecutor()
        val basicResult = BasicResult(callback, handler, executorService)
        executorService.execute {
            val isSuccessful = flushContextData(context)
            basicResult.setResult(
                if (isSuccessful) {
                    Result<Any?>(200, "")
                } else {
                    Result(
                        NetworkResponseException("Error Updating Context"),
                        ""
                    )
                }
            )
        }
        return basicResult
    }

    /** Sends Flybits Context Plugin data to the server if it is determined that
     * new Context data has been received.
     *
     * @param context  The state of the application.
     * @return true if context data uploaded to Flybits else returns false.
     * @throws NetworkResponseException when server call to flush the context data fails.
     */
    @JvmStatic
    fun flushContextData(context: Context): Boolean {
        if (context == null) {
            return false
        }
        var isSuccessful = false
        val dataPlugins =
            getDatabase(context).basicDataDao().allNotSent
        val jsonToSend = ContextUtilities.getContextData(dataPlugins)
        if (jsonToSend != null && jsonToSend.length > 2) {
            try {
                val result: Result<*> =
                    FlyAway.post<Any>(
                        context,
                        API_CONTEXT_DATA,
                        jsonToSend,
                        null,
                        "FlushUtilities.flushContext",
                        null
                    )
                if (result.status == RequestStatus.COMPLETED) {
                    dataPlugins.forEach {
                        it.isSent = true
                        getDatabase(context).basicDataDao().update(it)
                    }
                    isSuccessful = true
                } else {
                    appendTag(TAG_CM)
                        .e("Sending Flybits' ContextPlugin data to Server Failed.")
                }
            } catch (e: Exception) {
                throw NetworkResponseException()
            }
        }
        return isSuccessful
    }

    /**
     * Refresh the data that is associated to the [ContextPlugin], this may make a network
     * request, open a new activity/dialog, collect information from a sensor and/or anything else
     * that the [ContextPlugin] has be designed to do.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The [FlybitsContextPlugin] that should be refreshed.
     *
     *
     * Although this method seems a bit redundant, as it simply call the onRefresh of the Context
     * Plugin it has been written this way to be consistent with the other Flybits platforms
     */
    @JvmStatic
    fun refresh(mContext: Context, plugin: FlybitsContextPlugin) {
        coroutineScope.launch(exceptionHandlerBuilder("Error refresh the context plugins.")) {
            refreshSync(mContext, plugin)
        }
    }

    /**
     * Begin the [FlybitsContextPlugin] that is responsible for collecting Contextual Data about the
     * user or device only once. If the user is opted out the plugin will be saved and started if the user
     * opts back in. Executes on caller thread.
     * Please Note: This will only create worker to run it once.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The [FlybitsContextPlugin] that should be started and to run once.
     */
    @JvmStatic
    fun refreshSync(mContext: Context, plugin: FlybitsContextPlugin) {
        val user =
            CommonsDatabase.getDatabase(mContext).userDao().activeUser
        user?.let {
            if (it.isOptedIn) {
                appendTag(TAG_CM)
                    .d("Refreshing Context Plugins")
                plugin.onRefresh(mContext) //state will change here, insert into DAO after
            }
        }
        getDatabase(mContext).flybitsContextPluginDAO().insert(plugin)
    }

    /**
     * Refresh all the existing [FlybitsContextPlugin] to make them run once.
     *
     * @param mContext The Context of the Application.
     * @param callback The callback that indicates all plugins are refreshed.
     */
    @JvmStatic
    fun refreshAll(
        mContext: Context?,
        callback: BasicResultCallback?
    ) {
        mContext?.let {
            coroutineScope.launch(exceptionHandlerBuilder("Error refresh all the context plugins.")) {
                refreshAllSync(it, callback)
            }
        } ?: appendTag(TAG_CM).e("The Context parameter cannot be null.")
    }

    @JvmStatic
    private fun refreshAllSync(mContext: Context, callback: BasicResultCallback?) {
        val user =
            CommonsDatabase.getDatabase(mContext).userDao().activeUser
        val result = BasicResult(callback, Handler(Looper.getMainLooper()))
        user?.let {
            if (it.isOptedIn) {
                val flybitsContextPluginList =
                    getDatabase(mContext).flybitsContextPluginDAO().getAll()
                flybitsContextPluginList.forEach { pluginList ->
                    if (pluginList.isRunning) {
                        appendTag(TAG_CM)
                            .d("Refreshing Context Plugins ${pluginList.contextPluginRetriever.name}")
                        pluginList.onRefresh(mContext) //Run the onetime workers or services for plugins
                    }
                }
                result.setSuccess()
            } else {
                result.setFailed(UserOptedOutException("User has Opt out from Flybits"))
            }
        } ?: result.setFailed(NotConnectedException("No User has Connected to Flybits"))
    }

    /**
     * Begin the [FlybitsContextPlugin] that is responsible for collecting Contextual Data about the
     * user or device. If the user is opted out the plugin will be saved and started if the user
     * opts back in. Executes on worker thread.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The [FlybitsContextPlugin] that should be started.
     */
    @JvmStatic
    fun start(mContext: Context, plugin: FlybitsContextPlugin) {
        coroutineScope.launch(exceptionHandlerBuilder("Error starting the context plugins.")) {
            startSync(mContext, plugin)
        }
    }

    /**
     * Begin the [FlybitsContextPlugin] that is responsible for collecting Contextual Data about the
     * user or device. If the user is opted out the plugin will be saved and started if the user
     * opts back in. Executes on caller thread.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The [FlybitsContextPlugin] that should be started.
     */
    @JvmStatic
    fun startSync(mContext: Context, plugin: FlybitsContextPlugin) {
        val user =
            CommonsDatabase.getDatabase(mContext).userDao().activeUser
        if (plugin.contextPluginRetriever.genericSuperclass === FlybitsContextPluginsWorker::class.java) {
            user?.let {
                if (it.isOptedIn) {
                    appendTag(TAG_CM)
                        .d("Starting Context Plugins using workers")
                    plugin.onStart(mContext)
                }
            }
            getDatabase(mContext).flybitsContextPluginDAO().insert(plugin)
        } else if (plugin.contextPluginRetriever.genericSuperclass === FlybitsContextPluginService::class.java) { //Check if the service is added in the manifest.
            val intent = Intent(mContext, plugin.getService())
            if (ContextUtilities.isServiceDefined(mContext, intent)) {
                //state will change here, insert into DAO after
                if (user.isOptedIn) {
                    appendTag(TAG_CM)
                        .d("Starting Context Plugins using services")
                    plugin.onStart(mContext)
                }
                getDatabase(mContext).flybitsContextPluginDAO().insert(plugin)
            } else {
                appendTag(TAG_CM)
                    .e("Cannot start Context Plugin: In order to start the plugin you need to add the corresponding " + plugin.getService()!!.simpleName + " service in your manifest.")
            }
        } else {
            appendTag(TAG_CM).w(
                "This is unexpected since the FlybitsContextPlugin should either be of type FlybitsContextPluginService or FlybitsContextPluginsWorker"
            )
        }
    }

    /**
     * Resume a [FlybitsContextPlugin] that has previously been started.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The [FlybitsContextPlugin] that should be resumed.
     */
    @JvmStatic
    fun resume(mContext: Context, plugin: FlybitsContextPlugin) {
        coroutineScope.launch(exceptionHandlerBuilder("Error resuming the context plugins.")) {
            resumeSync(mContext, plugin)
        }
    }

    /**
     * Resume a [FlybitsContextPlugin] that has previously been started. Executes on
     * the caller thread
     *
     * @param mContext The Context of the Application.
     * @param plugin   The [FlybitsContextPlugin] that should be resumed.
     */
    @JvmStatic
    fun resumeSync(mContext: Context, plugin: FlybitsContextPlugin) {
        val user =
            CommonsDatabase.getDatabase(mContext).userDao().activeUser
        user?.let {
            resumeSync(mContext, plugin, it)
        }
    }

    /**
     * Resume a [FlybitsContextPlugin] that has previously been started. Executes on
     * the caller thread
     *
     * @param mContext The Context of the Application.
     * @param plugin   The [FlybitsContextPlugin] that should be resumed.
     * @param user     The currently authenticated user. It can be null to update/delete the plugins.
     */
    @JvmStatic
    fun resumeSync(
        mContext: Context,
        plugin: FlybitsContextPlugin,
        user: User?
    ) {
        if (plugin.contextPluginRetriever.genericSuperclass === FlybitsContextPluginsWorker::class.java) {
            user?.let {
                if (it.isOptedIn) {
                    appendTag(TAG_CM)
                        .d("Resume Context Plugins using Workers.")
                    plugin.onStart(mContext) //state will change here, insert into DAO after
                    getDatabase(mContext).flybitsContextPluginDAO().update(plugin)
                }
            }
        } else if (plugin.contextPluginRetriever.genericSuperclass === FlybitsContextPluginService::class.java) {
            val intent = Intent(mContext, plugin.getService())
            //Check if the service is added in the manifest
            if (ContextUtilities.isServiceDefined(mContext, intent)) {
                user?.let {
                    if (it.isOptedIn) {
                        appendTag(TAG_CM)
                            .d("Resume Context Plugins using Service.")
                        plugin.onStart(mContext) //state will change here, insert into DAO after
                        getDatabase(mContext).flybitsContextPluginDAO().update(plugin)
                    }
                }
            } else {
                getDatabase(mContext).flybitsContextPluginDAO().delete(plugin)
                appendTag(TAG_CM)
                    .e("Cannot start Context Plugin: In order to start the plugin you need to add the corresponding " + plugin.getService()!!.simpleName + " service in your manifest.")
            }
        } else {
            appendTag(TAG_CM).w(
                "This is unexpected since the FlybitsContextPlugin should either be of type FlybitsContextPluginService or FlybitsContextPluginsWorker"
            )
        }
    }

    /**
     * Stop the [FlybitsContextPlugin] that is responsible for collecting Contextual Data about the
     * user or device. Executes on worker thread.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The [FlybitsContextPlugin] that should be stopped.
     *
     *
     * Although this method seems a bit redundant, as it simply call the onStop of the Context
     * Plugin it has been written this way to be consistent with the other Flybits platforms
     */
    @JvmStatic
    fun stop(mContext: Context, plugin: FlybitsContextPlugin) {
        coroutineScope.launch(exceptionHandlerBuilder("Error stopping the context plugins.")) {
            stopSync(mContext, plugin)
        }
    }

    /**
     * Stop the [FlybitsContextPlugin] that is responsible for collecting Contextual Data about the
     * user or device. Executes on caller thread.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The [FlybitsContextPlugin] that should be stopped.
     *
     *
     * Although this method seems a bit redundant, as it simply call the onStop of the Context
     * Plugin it has been written this way to be consistent with the other Flybits platforms
     */
    @JvmStatic
    fun stopSync(mContext: Context, plugin: FlybitsContextPlugin) {
        appendTag(TAG_CM)
            .d("Stopping Context Plugins.")
        plugin.onStop(mContext) //state will change here, insert into DAO after
        getDatabase(mContext).flybitsContextPluginDAO().delete(plugin)
    }

    /**
     * Pause a running [FlybitsContextPlugin] so that it can be resumed at a later point.
     * Executes on worker thread.
     *
     * @param mContext             The application Context.
     * @param flybitsContextPlugin [FlybitsContextPlugin] to be paused.
     */
    @JvmStatic
    fun pause(
        mContext: Context,
        flybitsContextPlugin: FlybitsContextPlugin
    ) {
        coroutineScope.launch(exceptionHandlerBuilder("Error pausing the context plugins.")) {
            pauseSync(
                mContext,
                flybitsContextPlugin
            )
        }
    }

    /**
     * Pause a running [FlybitsContextPlugin] so that it can be resumed at a later point.
     * Executes on caller thread.
     *
     * @param mContext             The application Context.
     * @param flybitsContextPlugin [FlybitsContextPlugin] to be paused.
     */
    @JvmStatic
    fun pauseSync(
        mContext: Context,
        flybitsContextPlugin: FlybitsContextPlugin
    ) {
        appendTag(TAG_CM)
            .d("Pausing Context Plugins.")
        //Don't need to do anything here because it's already in the DAO and can be used during the resume
        flybitsContextPlugin.onStop(mContext) //state will change here, insert into DAO after
        getDatabase(mContext).flybitsContextPluginDAO().update(flybitsContextPlugin)
    }

    /**
     * Pause all running [FlybitsContextPlugin]s.
     *
     * @param mContext The Context of the Application.
     */
    @JvmStatic
    fun pauseAll(mContext: Context) {
        coroutineScope.launch(exceptionHandlerBuilder("Error pausing all the context plugins.")) {
            pauseAllSync(mContext)
        }
    }

    /**
     * Pause all running [FlybitsContextPlugin]s.
     *
     * @param mContext The Context of the Application.
     */
    @JvmStatic
    fun pauseAllSync(mContext: Context) {
        val flybitsContextPluginList =
            getDatabase(mContext).flybitsContextPluginDAO().getAll()
        flybitsContextPluginList.forEach { pluginList ->
            pauseSync(mContext, pluginList)
        }
    }

    /**
     * Resume all paused [FlybitsContextPlugin]s. Executes on worker thread.
     *
     * @param mContext The Context of the Application.
     */
    @JvmStatic
    fun resumeAll(mContext: Context) {
        coroutineScope.launch(exceptionHandlerBuilder("Error resuming all the context plugins.")) {
            resumeAllSync(mContext)
        }
    }

    /**
     * Resume all paused [FlybitsContextPlugin]s. Executes on caller thread.
     *
     * @param mContext The Context of the Application.
     */
    @JvmStatic
    fun resumeAllSync(mContext: Context) {
        val user =
            CommonsDatabase.getDatabase(mContext).userDao().activeUser
        user?.let {
            val flybitsContextPluginList =
                getDatabase(mContext).flybitsContextPluginDAO().getAll()
            flybitsContextPluginList.forEach { pluginList ->
                resumeSync(mContext, pluginList, it)
            }
        }
    }


    /**
     * To register the service [ContextRulesService] which retrieves the rules associated in
     * Flybits server
     *
     * @param mContext     The context of the application.
     * @param timeInSeconds Time to refresh in seconds.
     * @param timeInSecondsFlex Time interval to run the service after running first time in seconds.
     * @param unit Time unit defined for service to run.
     *
     */
    @Deprecated(
        "Deprecated in version 2.3.0, will be removed in version 4.0.0",
        ReplaceWith("This functionality will be removed from the SDK."),
        DeprecationLevel.WARNING
    )
    @JvmStatic
    fun registerForRules(
        mContext: Context,
        timeInSeconds: Long,
        timeInSecondsFlex: Long,
        unit: TimeUnit
    ) {
        val mGcmNetworkManager = GcmNetworkManager.getInstance(mContext)
        val task = PeriodicTask.Builder()
            .setTag("Context_Rules")
            .setUpdateCurrent(true)
            .setPersisted(true)
            .setPeriod(unit.toSeconds(timeInSeconds))
            .setService(ContextRulesService::class.java)
        mGcmNetworkManager.schedule(task.build())
        appendTag(TAG_CM)
            .d("Activated: Context_Rules, Time to Refresh: $timeInSeconds")
    }

    /**
     * To register the service [ContextPluginsService] which uploads the context plugins already associated
     * to Flybits server and insert it the local db.
     *
     * @param mContext           The context of the application.
     * @param listOfClass        HashMap of custom classes that should be started when Context Plugin is
     *                           enabled.
     * @param timeToRefreshInSec The time interval to upload Context data to the server.
     *
     */
    @Deprecated(
        "Deprecated in version 2.3.0, will be removed in version 4.0.0",
        ReplaceWith("This functionality will be removed from the SDK."),
        DeprecationLevel.WARNING
    )
    @JvmStatic
    fun registerForPluginUpdates(
        mContext: Context,
        listOfClass: HashMap<String?, String?>,
        timeToRefreshInSec: Long
    ) {
        val preferences =
            ContextScope.getContextPreferences(mContext).edit()
        preferences.putInt(ContextPluginsService.PREF_KEY_NUM_CPS, listOfClass.size)
        preferences.putLong(ContextPluginsService.PREF_KEY_REFRESH_CPS, timeToRefreshInSec)
        val bundle = Bundle()
        bundle.putInt("listOfItems", listOfClass.size)
        var i = 0
        listOfClass.forEach { (k, v) ->
            bundle.putString("item$i", k)
            preferences.putString("item$i", k)
            bundle.putString("class$i", v)
            bundle.putString("value$i", v)
            listOfClass.remove(k)
            i++
        }

        preferences.apply()
        val mGcmNetworkManager = GcmNetworkManager.getInstance(mContext)
        val oneTimeTask = OneoffTask.Builder()
            .setService(ContextPluginsService::class.java)
            .setTag("Context_Plugin_Once")
            .setExecutionWindow(0L, 30L)
            .setRequiredNetwork(Task.NETWORK_STATE_ANY)
            .build()
        mGcmNetworkManager.schedule(oneTimeTask)
        val task = PeriodicTask.Builder()
            .setExtras(bundle)
            .setTag("Context_Plugin")
            .setUpdateCurrent(true)
            .setPersisted(true)
            .setPeriod(timeToRefreshInSec)
            .setService(ContextPluginsService::class.java)
        mGcmNetworkManager.schedule(task.build())
        appendTag(TAG_CM)
            .d("Activated: Context_Plugin - Time to Refresh (in seconds): $timeToRefreshInSec")
    }


    /**
     * To register the service [ContextRulesService] which retrieves the rules associated in
     * Flybits server.
     *
     * @param mContext   The context of the application.
     * @param priority It defines the priority for uploading the context.
     * @param time  The refresh rate in seconds.
     * @param timeInSecondsFlex Time interval to run the service after running first time in seconds.
     * @param unit  The unit of the measure for time.
     */
    @Deprecated(
        "Deprecated in version 2.3.0, will be removed in version 4.0.0",
        ReplaceWith("This functionality will be removed from the SDK."),
        DeprecationLevel.WARNING
    )
    @JvmStatic
    fun registerUploadingContext(
        mContext: Context,
        priority: ContextPriority,
        time: Long,
        timeInSecondsFlex: Long,
        unit: TimeUnit
    ) {
        val bundle = Bundle()
        bundle.putInt(KEY_PRIORITY, priority.key)
        val mGcmNetworkManager = GcmNetworkManager.getInstance(mContext)
        val task = PeriodicTask.Builder().apply {
            this
                .setExtras(bundle)
                .setTag("Context_Uploading")
                .setUpdateCurrent(true)
                .setPersisted(true)
                .setRequiredNetwork(PeriodicTask.NETWORK_STATE_CONNECTED)
                .setPeriod(unit.toSeconds(time))
                .setService(ContextUploadingService::class.java)
        }
        mGcmNetworkManager.schedule(task.build())
        appendTag(TAG_CM)
            .d("Activated: Context_Uploading with Priority: " + priority.key + ", Time to Refresh: " + time)
    }

    /**
     * To register the worker [ContextUploadingWorker] which uploads the context plugins already associated
     *  to Flybits server and insert it the local db.
     *
     * @param time  The unit of the measure for time
     */
    @JvmStatic
    fun registerUploadingContext(time: Long) {
        scheduleWorker(time)
    }


    /**
     * To schedule the worker [ContextUploadingWorker] which uploads the context plugins data to
     * Flybits server.
     *
     * @param time  The unit of the measure for time
     */
    private fun scheduleWorker(time: Long) {
        val constraintsNetwork = Constraints.Builder()
            .setRequiredNetworkType(NetworkType.CONNECTED)
            .build()

        val workRequest = PeriodicWorkRequest
            .Builder(ContextUploadingWorker::class.java, time, TimeUnit.MINUTES)
            .setConstraints(constraintsNetwork)
            .build()
        WorkManager.getInstance().enqueueUniquePeriodicWork(
            ContextUploadingWorker::class.java.simpleName
            , ExistingPeriodicWorkPolicy.REPLACE
            , workRequest
        )
        appendTag(TAG_CM)
            .d("Activated: Context_Uploading , Time to Refresh: $time")
    }


    /**
     * To cancel the service [ContextRulesService] which retrieves the rules associated in
     * Flybits server
     *
     * @param mContext  The context of the application.
     *
     */
    @Deprecated(
        "Deprecated in version 2.3.0, will be removed in version 4.0.0",
        ReplaceWith("This functionality will be removed from the SDK."),
        DeprecationLevel.WARNING
    )
    @JvmStatic
    fun unregisterFromRuleCollection(mContext: Context) {
        val intent = Intent(mContext, ContextRulesService::class.java)
        if (ContextUtilities.isServiceDefined(mContext, intent)) {
            val mGcmNetworkManager = GcmNetworkManager.getInstance(mContext)
            try {
                mGcmNetworkManager.cancelTask("Context_Rules", ContextRulesService::class.java)
                appendTag(TAG_CM)
                    .d("UnActivated: Context_Rules")
            } catch (ex: IllegalArgumentException) {
                exception(
                    "ContextManager.unregisterFromRuleCollection",
                    ex
                )
            }
        }
        appendTag(TAG_CM)
            .d("Unregistered service for retrieving rules.")
    }

    /**
     * To cancel the service [ContextUploadingService] which uploads the context plugins data to
     * Flybits server
     *
     * @param mContext  The context of the application.
     */
    @Deprecated(
        "Deprecated in version 2.3.0, will be removed in version 4.0.0",
        ReplaceWith("Since [GcmNetworkManager] API deprecated, instead using [WorkManager] API."),
        DeprecationLevel.WARNING
    )
    @JvmStatic
    fun unregisterUploadingContext(mContext: Context) {
        val intent = Intent(mContext, ContextUploadingService::class.java)
        if (ContextUtilities.isServiceDefined(mContext, intent)) {
            val mGcmNetworkManager = GcmNetworkManager.getInstance(mContext)
            try {
                mGcmNetworkManager.cancelTask(
                    "Context_Uploading",
                    ContextUploadingService::class.java
                )
                appendTag(TAG_CM)
                    .d("UnActivated: Context_Uploading")
            } catch (ex: IllegalArgumentException) {
                exception(
                    "ContextManager.unregisterUploadingContext",
                    ex
                )
            }
        }
        appendTag(TAG_CM)
            .d("Unregistered service for UploadingContext.")
    }

    /**
     * To cancel the worker [ContextUploadingWorker] which uploads the context plugins data
     * to Flybits Server.
     */
    @JvmStatic
    fun unregisterUploadingContext() {
        WorkManager.getInstance().cancelUniqueWork(ContextUploadingWorker::class.java.simpleName)
        appendTag(TAG_CM)
            .d("ContextManager.unregisterPluginContext Completed")
    }

    /**
     * To cancel the service [ContextPluginsService] which uploads the context plugins already associated
     * to Flybits server and insert it the local db.
     *
     * @param mContext  The context of the application.
     */
    @Deprecated(
        "Deprecated in version 2.3.0, will be removed in version 4.0.0",
        ReplaceWith("Since [GcmNetworkManager] API deprecated, instead using [WorkManager] API."),
        DeprecationLevel.WARNING
    )
    @JvmStatic
    fun unregisterPluginContext(mContext: Context) {
        val intent = Intent(mContext, ContextPluginsService::class.java)
        if (ContextUtilities.isServiceDefined(mContext, intent)) {
            val mGcmNetworkManager = GcmNetworkManager.getInstance(mContext)
            try {
                mGcmNetworkManager.cancelTask("Context_Plugin", ContextPluginsService::class.java)
                appendTag(TAG_CM)
                    .d("UnActivated: Context_Plugin")
            } catch (ex: IllegalArgumentException) {
                exception(
                    "ContextManager.unregisterPluginContext",
                    ex
                )
            }
        }
        appendTag(TAG_CM)
            .d("Unregistered service for uploading Context Plugins.")
    }

    @JvmStatic
    private fun <T : ContextData> getDataPrivate(
        context: Context,
        pluginID: String,
        subclass: Class<T>,
        callback: ObjectResultCallback<T>?
    ): ObjectResult<T>? {
        val handler = Handler(Looper.getMainLooper())
        val executorService =
            Executors.newSingleThreadExecutor()
        val objectResult = ObjectResult(callback, handler, executorService)
        executorService.execute {
            val dataFromCursor =
                getDatabase(context).basicDataDao()[pluginID]
            if (dataFromCursor != null && dataFromCursor.valueAsString != null) {
                try {
                    val data = subclass.newInstance() as T
                    data.fromJson(dataFromCursor.valueAsString)
                    data.time = dataFromCursor.timestamp
                    val result =
                        Result<T>(200, "")
                    result.setResponse(data)
                    objectResult.setResult(result)
                } catch (e: Exception) {
                    objectResult.setResult(
                        Result(
                            FlybitsException(e.message),
                            ""
                        )
                    )
                    Logger.e("Exception on ContextManager.getData")
                }
            }
        }
        return objectResult
    }
}