package com.flybits.concierge.smartrewards.viewproviders

import android.content.Context
import android.graphics.drawable.Drawable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.TextView
import com.bumptech.glide.Glide
import com.bumptech.glide.request.RequestOptions
import com.bumptech.glide.request.target.CustomTarget
import com.bumptech.glide.request.transition.Transition
import com.flybits.android.kernel.ContentAnalytics
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.FlybitsViewProvider
import com.flybits.concierge.smartrewards.R
import com.flybits.concierge.smartrewards.model.Confirmation
import com.flybits.concierge.smartrewards.views.RoundedCornerBorderButton
import com.flybits.concierge.viewholders.FlybitsViewHolder

/**
 * Provides UI and business logic for the [Confirmation] content instance.
 *
 * Use `ConfirmationViewProvider(Context)` to instantiate.
 */
class ConfirmationViewProvider internal constructor(private var contentAnalytics: ContentAnalytics?) :
    FlybitsViewProvider<Confirmation> {

    /**
     * Main constructor to be used for instantiating the [ConfirmationViewProvider].
     *
     * @param context The application's [Context].
     *
     */
    constructor(context: Context) : this(ContentAnalytics(context))

    @Deprecated(
        "A new constructor has been made available which will allow for analytics support. Deprecated in version 3.0.0, will be removed in version 4.0.0",
        ReplaceWith("ConfirmationViewProvider(applicationContext)")
    )
    constructor() : this(null)

    override fun getClassType(): Class<Confirmation> {
        return Confirmation::class.java
    }

    override fun getContentType(): String {
        return "ctn-flybits-reward-confirmation-card"
    }

    override fun onCreateViewHolder(parent: ViewGroup): FlybitsViewHolder<Confirmation> {
        return ConfirmationViewHolder(
            LayoutInflater.from(parent.context)
                .inflate(R.layout.smartrewards_item_template_reward, parent, false),
            contentAnalytics
        )

    }

    override fun onPushOpened(model: Confirmation, flybitsNavigator: FlybitsNavigator) {

    }

    companion object {
        const val CONFIRMATION_CARD_TAG = "ConfirmationCardTag"
    }

    // View Holder for the Benefit Card
    class ConfirmationViewHolder(val view: View, val contentAnalytics: ContentAnalytics?) :
        FlybitsViewHolder<Confirmation>(view) {

        private val headerView: TextView =
            view.findViewById(R.id.smartrewards_item_template_reward_header)
        private val titleView: TextView =
            view.findViewById(R.id.smartrewards_item_template_reward_title)
        private val descriptionView: TextView =
            view.findViewById(R.id.smartrewards_item_template_reward_description)
        private val iconView: ImageView =
            view.findViewById(R.id.smartrewards_item_template_reward_icon)
        private val ctaContainerView: View =
            view.findViewById(R.id.smartrewards_item_template_reward_cta_container)
        private val resultStatusView: TextView =
            view.findViewById(R.id.smartrewards_item_template_reward_result_status)
        private val resultButton: RoundedCornerBorderButton =
            view.findViewById(R.id.smartrewards_item_template_reward_result_btn)

        private val backgroundView: ImageView =
            view.findViewById(R.id.smartrewards_item_template_reward_background_image)
        private val gradientView: View =
            view.findViewById(R.id.smartrewards_item_template_reward_gradient)

        override fun bindData(data: Confirmation, flybitsNavigator: FlybitsNavigator) {

            data.backgroundColor?.let {
                gradientView.setBackgroundColor(data.getBackgroundColor())
            }

            data.gradientDrawable?.let {
                gradientView.background = it
            }

            headerView.text = data.header?.value ?: ""
            titleView.text = data.title?.value ?: ""
            descriptionView.text = data.explanation?.value ?: ""
            resultStatusView.text = data.rewardStatus ?: ""

            if (!data.icon.isNullOrEmpty()) {
                iconView.visibility = View.GONE
                Glide.with(iconView.context).load(data.icon)
                    .into(object : CustomTarget<Drawable>() {
                        override fun onLoadFailed(errorDrawable: Drawable?) {
                            Logger.appendTag(CONFIRMATION_CARD_TAG)
                                .e(iconView.context.getString(R.string.smartrewards_image_error))
                            iconView.visibility = View.GONE
                        }

                        override fun onLoadCleared(placeholder: Drawable?) {}
                        override fun onResourceReady(
                            resource: Drawable,
                            transition: Transition<in Drawable>?
                        ) {
                            iconView.visibility = View.VISIBLE
                            iconView.setImageDrawable(resource)
                        }
                    })
            }

            ctaContainerView.visibility = View.GONE

            // Build Background
            gradientView.alpha = 1.0f
            backgroundView.visibility = View.GONE

            if (!data.backgroundImage.isNullOrEmpty()) {
                Glide.with(iconView.context).load(data.backgroundImage)
                    .apply(RequestOptions().centerCrop())
                    .into(object : CustomTarget<Drawable>() {
                        override fun onLoadFailed(errorDrawable: Drawable?) {
                            Logger.appendTag(CONFIRMATION_CARD_TAG)
                                .e(iconView.context.getString(R.string.smartrewards_image_error))
                            gradientView.alpha = 1.0f
                            backgroundView.visibility = View.GONE
                        }

                        override fun onLoadCleared(placeholder: Drawable?) {}
                        override fun onResourceReady(
                            resource: Drawable,
                            transition: Transition<in Drawable>?
                        ) {
                            backgroundView.visibility = View.VISIBLE
                            gradientView.alpha = 0.6f
                            backgroundView.setImageDrawable(resource)
                        }
                    })
            }

            resultButton.visibility = View.INVISIBLE
            if (data.primaryButton?.label != null) {
                resultButton.visibility = View.VISIBLE

                resultButton.setText(data.primaryButton.label.value)
                resultButton.setColor(data.primaryButton.getColor())

                resultButton.setOnClickListener {
                    contentAnalytics?.trackEngaged(data.content)
                    contentAnalytics?.trackFulfilled(data.content) // Full-filled analytics when clicked Learn More.
                    if (data.primaryButton.actionType ?: "" == "URL") {
                        flybitsNavigator.openUrl(data.primaryButton.action)
                    }
                }
            }

        }
    }
}