package com.flybits.concierge.smartrewards.viewproviders

import android.content.Context
import android.graphics.drawable.Drawable
import android.os.Bundle
import android.support.v7.widget.LinearLayoutManager
import android.support.v7.widget.PagerSnapHelper
import android.support.v7.widget.RecyclerView
import android.text.TextUtils
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.TextView
import com.bumptech.glide.Glide
import com.bumptech.glide.request.RequestOptions
import com.bumptech.glide.request.target.CustomTarget
import com.bumptech.glide.request.transition.Transition
import com.flybits.android.kernel.ContentAnalytics
import com.flybits.commons.library.logging.Logger
import com.flybits.concierge.FlybitsNavigator
import com.flybits.concierge.FlybitsViewProvider
import com.flybits.concierge.UserActionHandler
import com.flybits.concierge.smartrewards.R
import com.flybits.concierge.smartrewards.activities.OfferActivity
import com.flybits.concierge.smartrewards.model.Offer
import com.flybits.concierge.smartrewards.model.Offers
import com.flybits.concierge.smartrewards.useractionhandlers.OfferUserActionHandler
import com.flybits.concierge.smartrewards.utils.ViewHelper
import com.flybits.concierge.viewholders.FlybitsViewHolder

/**
 * Provides UI and business logic for the [Offers] content instance.
 *
 * Use `OffersViewProvider(Context)` to instantiate.
 */
class OffersViewProvider internal constructor(
    private val contentAnalytics: ContentAnalytics?, val userActionHandler: UserActionHandler<Offer>
) : FlybitsViewProvider<Offers> {

    /**
     * Constructor used for instantiating the [OffersViewProvider] with the default [UserActionHandler] for [Offers].
     *
     * @param context The application's [Context].
     */
    constructor(context: Context) : this(ContentAnalytics(context), OfferUserActionHandler())

    /**
     * Constructor used for instantiating the [OffersViewProvider] with a specific [OfferUserActionHandler].
     *
     * @param context The application's [Context].
     *
     * @param userActionHandler The [UserActionHandler] which will respond to user actions in the UI.
     * When overriding only a subset of the default handling logic, extend and override [OfferUserActionHandler].
     */
    constructor(context: Context, userActionHandler: UserActionHandler<Offer>)
            : this(ContentAnalytics(context), userActionHandler)

    @Deprecated(
        "A new constructor has been made available which will allow for analytics support. deprecated in version 3.0.0, will be removed in version 4.0.0",
        ReplaceWith(expression = "OffersViewProvider(applicationContext)")
    )
    constructor() : this(null, OfferUserActionHandler())

    override fun getClassType(): Class<Offers> {
        return Offers::class.java
    }

    override fun getContentType(): String {
        return CONTENT_TYPE
    }

    override fun onCreateViewHolder(parent: ViewGroup): FlybitsViewHolder<Offers> {
        val view = LayoutInflater.from(parent.context)
            .inflate(R.layout.smartrewards_item_template_offers, parent, false)
        return OffersViewHolder(view, contentAnalytics, userActionHandler)
    }

    override fun onPushOpened(model: Offers, flybitsNavigator: FlybitsNavigator) {
        //Display offer if only one is present in the Offers model, otherwise if multiple are present then scroll
        if (model.offers != null && model.offers.size == 1) {
            val bundle = Bundle()
            bundle.putParcelable(OfferActivity.EXTRA_OFFER_DETAILS, model.offers[0])
            bundle.putParcelable(OfferActivity.EXTRA_USER_ACTION_HANDLER, userActionHandler)
            flybitsNavigator.openActivity(OfferActivity::class.java, bundle)
        }
    }

    companion object {
        const val OFFERS_CARD_TAG = "OffersCardTag"
        const val CONTENT_TYPE = "ctn-flybits-offers-card"

        private fun configureView(
            data: Offer,
            offerView: View,
            userActionHandler: UserActionHandler<Offer>,
            flybitsNavigator: FlybitsNavigator,
            contentAnalytics: ContentAnalytics?
        ) {
            val bannerView: ImageView =
                offerView.findViewById(R.id.smartrewards_item_template_offer_banner)
            val headerView: TextView =
                offerView.findViewById(R.id.smartrewards_item_template_offer_header)
            val titleView: TextView =
                offerView.findViewById(R.id.smartrewards_item_template_offer_title)
            val subtitleView: TextView =
                offerView.findViewById(R.id.smartrewards_item_template_offer_subtitle)

            if (!TextUtils.isEmpty(data.banner)) { // banner is optional
                Glide.with(bannerView.context).load(data.banner)
                    .apply(RequestOptions().centerCrop()).into(object : CustomTarget<Drawable>() {
                        override fun onLoadFailed(errorDrawable: Drawable?) {
                            Logger.appendTag(OFFERS_CARD_TAG)
                                .e(bannerView.context.getString(R.string.smartrewards_image_error))
                            bannerView.setImageResource(android.R.drawable.ic_menu_gallery)
                        }

                        override fun onLoadCleared(placeholder: Drawable?) {}
                        override fun onResourceReady(
                            resource: Drawable,
                            transition: Transition<in Drawable>?
                        ) {
                            bannerView.setImageDrawable(resource)
                        }
                    })
            } else {
                bannerView.setImageResource(android.R.drawable.ic_menu_gallery)
            }

            headerView.text = data.header.value
            titleView.text = data.title.value
            subtitleView.text = data.subtitle.value

            offerView.setOnClickListener {
                contentAnalytics?.trackEngaged(data.content)
                userActionHandler.onUserAction(
                    OfferUserActionHandler.TAPPED_TILE,
                    data,
                    flybitsNavigator
                )
            }
        }
    }

    class OffersViewHolder(
        val view: View,
        val contentAnalytics: ContentAnalytics?,
        val userActionHandler: UserActionHandler<Offer>
    ) : FlybitsViewHolder<Offers>(view) {
        private val recycler: RecyclerView =
            view.findViewById(R.id.smartrewards_item_template_offers_recyclerView)
        private val single: View = view.findViewById(R.id.smartrewards_item_template_offers_single)
        private val emptyView: View =
            view.findViewById(R.id.smartrewards_item_template_offers_emptyTxt)

        override fun bindData(data: Offers, flybitsNavigator: FlybitsNavigator) {
            recycler.visibility = View.GONE
            single.visibility = View.GONE
            emptyView.visibility = View.GONE

            data.offers.forEach { it.content = data.content } //set child content to parent content

            when (data.offers.size) {
                0 -> {
                    emptyView.visibility = View.VISIBLE
                }
                1 -> {
                    single.visibility = View.VISIBLE
                    configureView(
                        data.offers[0],
                        view,
                        userActionHandler,
                        flybitsNavigator,
                        contentAnalytics
                    )
                }
                else -> {
                    recycler.visibility = View.VISIBLE
                    recycler.layoutManager =
                        LinearLayoutManager(recycler.context, LinearLayoutManager.HORIZONTAL, false)
                    recycler.adapter = OffersRecyclerAdapter(
                        data.offers,
                        flybitsNavigator,
                        userActionHandler,
                        contentAnalytics
                    )
                    recycler.onFlingListener =
                        null // need to unset the fling listener. platform bug
                    val snapHelper = PagerSnapHelper()
                    snapHelper.attachToRecyclerView(recycler)
                }
            }
        }
    }

    class OffersRecyclerAdapter(
        private val offers: List<Offer>,
        private val flybitsNavigator: FlybitsNavigator,
        private val userActionHandler: UserActionHandler<Offer>,
        private val contentAnalytics: ContentAnalytics?
    ) : RecyclerView.Adapter<OffersRecyclerAdapter.ViewHolder>() {

        class ViewHolder(val view: View) : RecyclerView.ViewHolder(view) {
            init {
                view.layoutParams.width = ViewHelper.cardPeekWidth(view.context)
            }
        }

        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ViewHolder {
            return ViewHolder(
                LayoutInflater.from(parent.context)
                    .inflate(R.layout.smartrewards_item_template_offer, parent, false)
            )
        }

        override fun getItemCount(): Int {
            return offers.size
        }

        override fun onBindViewHolder(holder: ViewHolder, position: Int) {
            configureView(
                offers[position],
                holder.view,
                userActionHandler,
                flybitsNavigator,
                contentAnalytics
            )
        }
    }
}