package com.flybits.commons.library.utils.filters;

/**
 * The {@code RegisterOptions} class is used to define all the user properties that are
 * optional/mandatory whenever a user registers to the Flybits system.
 *
 * @since 2.3.4
*/
public class RegisterOptions {

    private String email;
    private String password;
    private String firstName;
    private String lastName;
    private boolean retrieveRememberMeToken;
    private boolean isJWTSet;

    private RegisterOptions(Builder.Options builder){


        if (builder.email != null){
            this.email      = builder.email;
        }
        if (builder.password != null){
            this.password      = builder.password;
        }
        if (builder.firstName != null){
            this.firstName      = builder.firstName;
        }
        if (builder.lastName != null){
            this.lastName      = builder.lastName;
        }
        this.retrieveRememberMeToken        = builder.retrieveRememberMeToken;
        this.isJWTSet                       = builder.isJWTSet;
    }

    /**
     * Gets the JSON String representation of the body that is used for the Request object when
     * registering a new user to the Flybits system.
     *
     * @return The JSON String representation of the request body to be used when registering a new
     * user.
     */

    public String getRequest(){
        String rawJson = "{";

        if (email != null && password != null){

            rawJson += ("\"email\": \"" +email+ "\",");
            rawJson += ("\"password\": \"" +password+ "\"");

        }

        if (isJWTSet){
            rawJson += (", \"includeCredentialsJwt\" : true");
        }

        if (firstName != null){
            rawJson += (", \"firstName\" : \"" + firstName + "\"");
        }
        if (lastName != null){
            rawJson += (", \"lastName\" : \"" + lastName + "\"");
        }
        rawJson += "}";
        return rawJson;
    }

    /**
     * Indicates whether or not the user has indicated that they would like to keep the user logged
     * in until they explicitly log out.
     *
     * @return true if the application has indicated that they would like to keep the user logged in
     * to the application until the application explicitly logs out, false otherwise.
     */
    public boolean isRetrieveRememberMeToken(){
        return retrieveRememberMeToken;
    }

    /**
     * The {@code Builder} class defines the mandatory parameters that are needed for registering
     * users. Currently, there is only one mandatory method
     * {@link RegisterOptions.Builder#setUserCredentials(String, String)}.
     */

    public static class Builder{

        private String email;
        private String password;

        /**
         * Constructor used to create the initialize the {@code Builder} object.
         */
        public Builder(){}

        /**
         * Set the user's email and password. These are the two mandatory fields needed to create a
         * Flybits user.
         *
         * @param email The email address registered to the user. This email must be unique.
         * @param password The password associated to the {@code email}.
         *
         * @return A {@link RegisterOptions.Builder} object
         * where additional register settings can be set.
         */
        public Options setUserCredentials(String email, String password){
            this.email      = email;
            this.password   = password;
            return new Options(this);
        }

        /**
         * The {@code Options} used class which defines all optional settings that can be added to
         * the Register FilterOptions including:
         * <ul>
         *     <li>Disabling RememberMe Options</li>
         *     <li>Disable JWT Token</li>
         *     <li>Setting User Information such as First/Last Name</li>
         * </ul>
         */
        public static class Options{

            private String email;
            private String password;
            private String firstName;
            private String lastName;
            private boolean retrieveRememberMeToken;
            private boolean isJWTSet;

            /**
             * Constructor used to create the initialize the {@code Options} object.
             * @param builder The {@link Builder} which contains information about the user's
             *                credentials.
             */
            public Options(RegisterOptions.Builder builder){

                if (builder.email != null){
                    email   = builder.email;
                }

                if (builder.password != null){
                    password   = builder.password;
                }

                retrieveRememberMeToken = true;
                isJWTSet                = true;
            }

            /**
             * Build the {@link RegisterOptions} object to be used for constructing the POST request
             * for registering a user.
             *
             * @return {@link RegisterOptions} object which can be referenced by the SDK to get
             * specific information about the defined Register mechanism.
             */
            public RegisterOptions build() {

                return new RegisterOptions(this);
            }

            /**
             * Disables the retrieval of the remember me token attribute. If this method is set the
             * user's login session will only last a small amount of time before the user has to
             * login to recreate their session.
             *
             * @return A {@link RegisterOptions.Builder.Options}
             * object where additional register settings can be set.
             */
            public Options disableRememberMe(){
                this.retrieveRememberMeToken    = false;
                return this;
            }

            /**
             * Disables the retrieval of the JWT token. If this method is set the application will
             * not be able to get context information about the user as well push related
             * notifications.
             *
             * @return A {@link RegisterOptions.Builder.Options}
             * object where additional register settings can be set.
             */
            public Options disableJWT(){
                this.isJWTSet               = false;
                return this;
            }

            /**
             * Sets the user's personal information with their account including their first and
             * last names.
             *
             * @param firstName The first name of the user.
             * @param lastName The last name of the user.
             *
             * @return A {@link RegisterOptions.Builder.Options}
             * object where additional register settings can be set.
             */
            public Options setUserInformation(String firstName, String lastName) {
                this.firstName = firstName;
                this.lastName = lastName;
                return this;
            }
        }
    }
}
