package com.flybits.commons.library.models;

import android.os.Parcel;
import android.os.Parcelable;

/**
 * This class represents a {@code User} that can be found within this application environment. A
 * {@code User} is defined by their profile attributes as well an any additional metadata associated
 * to the {@code User} that may be specific to the application.
 *
 * @since 1.0.0
 */
public class User implements Parcelable {

    /**
     * The gender options for the user
     */
    public enum Gender {
        /**
         * Indicates the gender is not known.
         */
        UNKNOWN("unknown"),

        /**
         * Indicates the gender as male.
         */
        MALE("male"),

        /**
         * Indicates the gender as female.
         */
        FEMALE("female");

        private final String key;

        /**
         * Constructor that defines the key for each {@code Gender} option.
         *
         * @param key the string value representing each {@code Gender} option.
         */
        Gender(String key) {
            this.key = key;
        }

        /**
         * Get the string representation for the {@code Gender} option.
         *
         * @return string representation of the {@code Gender} option.
         */
        public String getKey() {
            return this.key;
        }

        /**
         * Get the Gender enum value corresponding to an String representation.
         *
         * @param key the string representation of the {@code Gender} enum.
         *
         * @return The {@code Gender} enum for the String representation.
         */
        public static Gender fromKey(String key) {
            for(Gender type : Gender.values()) {
                if(type.getKey().equals( key)) {
                    return type;
                }
            }
            return null;
        }
    }

    /**
     * The date of birth of the {@code User} represented by {@link DateOfBirth}.
     */
    public DateOfBirth dateOfBirth;

    /**
     * The email of the {@code User}.
     */
    public String email;

    /**
     * The first name of the {@code User}.
     */
    public String firstName;

    /**
     * The gender of the user represented by {@link User.Gender}.
     */
    public Gender gender = Gender.UNKNOWN;

    private String icon;

    /**
     * The unique identifier for the {@code User's} avatar. This may be null.
     */
    public String iconId;

    /**
     * The unique identifier of the {@code User}.
     */
    public String id;

    /**
     * Token that is used to unique identify a User/Device/Application combination which is useful
     * to quickly authenticate that a user is allowed to access particular SDK methods that are
     * available to them.
     */
    public String jwtToken;

    /**
     * The last name of the {@code User}.
     */
    public String lastName;

    /**
     * Default constructor that creates an empty {@code User} object.
     */
    public User(){}

    /**
     * Constructor used for un-flattening a {@code User} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code User} parcel.
     */
    public User(Parcel in){

        id                  = in.readString();
        firstName           = in.readString();
        lastName            = in.readString();
        icon                = in.readString();
        iconId              = in.readString();
        email               = in.readString();
        jwtToken            = in.readString();

        gender          = Gender.fromKey(in.readString());
        dateOfBirth     = in.readParcelable(getClass().getClassLoader());
    }

    /**
     * Get the icon url of the {@code User}.
     *
     * @return the icon url of the {@code User}. If no icon url is available then null will be
     * returned.
     */
    public String getIcon(){
        return icon;
    }

    /**
     * Set the icon url of the {@code User}.
     *
     * @param value the icon url value.
     */
    public void setIcon(String value){
        icon = value;
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    public int describeContents() {
        return 0;
    }

    @Override
    public boolean equals(Object o) {

        if (!(o instanceof User))
            return false;

        User another = (User) o;
        return (another.id.equals(id));
    }

    /**
     * Flatten this {@code User} into a Parcel.
     *
     * @param out The Parcel in which the {@code User} object should be written.
     * @param flags Additional flags about how the DateOfBirth object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    public void writeToParcel(Parcel out, int flags) {

        out.writeString(id);
        out.writeString(firstName);
        out.writeString(lastName);
        out.writeString(icon);
        out.writeString(iconId);
        out.writeString(email);
        out.writeString(jwtToken);
        out.writeString(gender.getKey());
        out.writeParcelable(dateOfBirth, flags);
    }

    /**
     * Parcelable.Creator that instantiates {@code User} objects
     */
    public static final Creator<User> CREATOR = new Creator<User>(){
        public User createFromParcel(Parcel in){
            return new User(in);
        }

        public User[] newArray(int size){
            return new User[size];
        }
    };
}
