package com.flybits.commons.library.models;

import android.content.Context;
import android.os.Parcel;
import android.os.Parcelable;


/**
 * A class that represents the information about the user's device. When a user's device is
 * registered with the Flybits Core a unique {@link Device#id} is generated by the Flybits Core and
 * the physical device identifier is then associated with this unique {@link Device#id}. This
 * provides an extra level of security as the physical identifier for the device is not exposed via
 * any APIs.
 *
 * @since 1.0.0
 */
public class Device implements Parcelable {

    /**
     * Indicates the Operating System of the physical device.
     */
    public enum Type {

        /**
         * Indicates that an unknown Operating System was associated to this physical device.
         */
        UNKNOWN("unknown"),

        /**
         * Indicates that an Android Operating System was associated to this physical device.
         */
        Android("android"),

        /**
         * Indicates that an iOS Operating System was associated to this physical device.
         */
        iOS("iOS");

        private final String key;

        /**
         * Constructor that defines the key for the each {@code Type} option.
         *
         * @param key the String value representing each {@code Type} option.
         */
        Type(String key) {
            this.key = key;
        }

        /**
         * Get the String representation for the {@code Type} option.
         *
         * @return String representation of the {@code Type} option.
         */
        public String getKey() {
            return this.key;
        }

        /**
         * Get the {@code Type} enum value corresponding to an String representation.
         *
         * @param key the String representation of the {@code Type} enum.
         *
         * @return The {@code Type} enum for the String representation.
         */
        public static Type fromKey(String key) {
            for(Type type : Type.values()) {
                if(type.getKey().equals( key)) {
                    return type;
                }
            }
            return UNKNOWN;
        }
    }

    /**
     * The time, in milliseconds, when this device was first entered into the Flybits System.
     */
    public long createdAt;

    /**
     * The unique identifier for the physical device.
     */
    public String id;

    /**
     * The unique physical identifier for the device. This can also be obtained using the
     * {@link com.flybits.commons.library.utils.Utilities#getDeviceID(Context)} method.
     */
    public String physicalId;

    /**
     * Indicates the Operating System of the physical device as specified with {@link Type}.
     */
    public Type type;


    /**
     * Initial Empty Constructor.
     */
    public Device(){}

    /**
     * Constructor used for un-flattening a {@code Device} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Device} parcel.
     */
    public Device(Parcel in){
        createdAt   = in.readLong();
        id          = in.readString();
        physicalId  = in.readString();
        type        = Type.fromKey(in.readString());
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    public int describeContents() {
        return 0;
    }

    @Override
    public boolean equals(Object o) {

        if (o instanceof String){
            return physicalId.equalsIgnoreCase(o.toString());
        }

        if (!(o instanceof Device))
            return false;

        Device another = (Device) o;
        return (another.physicalId.equals(physicalId));
    }

    /**
     * Flatten this {@code Device} into a Parcel.
     *
     * @param out The Parcel in which the {@code Device} object should be written.
     * @param flags Additional flags about how the DateOfBirth object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    public void writeToParcel(Parcel out, int flags) {

        out.writeLong(createdAt);
        out.writeString(id);
        out.writeString(physicalId);
        out.writeString(type == null ? User.Gender.UNKNOWN.getKey() : type.getKey());
    }

    /**
     * Parcelable.Creator that instantiates {@code Device} objects
     */
    public static final Creator<Device> CREATOR = new Creator<Device>() {
        public Device createFromParcel(Parcel in) {
            return new Device(in);
        }

        public Device[] newArray(int size) {
            return new Device[size];
        }
    };
}
