package com.flybits.commons.library;

import android.content.Context;
import android.content.SharedPreferences;
import android.support.annotation.NonNull;

import com.flybits.commons.library.models.JWTToken;
import com.flybits.commons.library.utils.Utilities;

import java.util.ArrayList;

import static android.content.Context.MODE_PRIVATE;

/**
 * This class is used to get and set all variables that can be shared across the various Flybits'
 * Android SDKs. Currently the items that can be shared between the SDKs are as follows;
 * <ul>
 *     <li>Language Information</li>
 *     <li>Device Identifier</li>
 *     <li>User Identifier</li>
 *     <li>JWT</li>
 *     <li>Indicator for whether or not the user is authenticated</li>
 * </ul>
 */
public class SharedElements {

    /**
     * Get the ArrayList representation of the language codes set for this application.
     *
     * @param context The context of the application.
     * @return The ArrayList representation of the language codes set for this application.
     */
    public static ArrayList<String> getEnabledLanguagesAsArray(Context context){
        SharedPreferences preferences = getPreferences(context);
        String languageCodes = preferences.getString(FlyingConstants.PREF_LANGUAGE_CODES, FlyingConstants.PREF_DEFAULT_EMPTY_STRING);
        return Utilities.convertLocalizationStringToList(languageCodes);
    }

    /**
     * Get the String representation of the language codes set for this application.
     *
     * @param context The context of the application.
     * @return The String representation of the language codes set for this application.
     */
    public static String getEnabledLanguagesAsString(Context context){
        SharedPreferences preferences = getPreferences(context);
        return preferences.getString(FlyingConstants.PREF_LANGUAGE_CODES, FlyingConstants.PREF_DEFAULT_EMPTY_STRING);
    }

    /**
     * Get the previously saved {@link com.flybits.commons.library.models.Device#id}.
     *
     * @param context The context of the application.
     * @return The saved {@link com.flybits.commons.library.models.Device#id} or "" if no
     * {@link com.flybits.commons.library.models.Device#id} is saved.
     */
    public static String getDeviceID(Context context){
        SharedPreferences preferences   = getPreferences(context);
        return preferences.getString(FlyingConstants.PREF_DEVICE_ID, FlyingConstants.PREF_DEFAULT_EMPTY_STRING);
    }

    /**
     * Get the previously saved {@link JWTToken#payload}.
     *
     * @param context The context of the application.
     * @return The saved {@link JWTToken#payload} or "" if no {@link JWTToken#payload} is saved.
     */
    public static String getSavedJWTToken(Context context){
        SharedPreferences preferences   = getPreferences(context);
        return preferences.getString(FlyingConstants.PREF_JWT_TOKEN, FlyingConstants.PREF_DEFAULT_EMPTY_STRING);
    }

    /**
     * Get the previously saved {@link com.flybits.commons.library.models.User#id}.
     *
     * @param context The context of the application.
     * @return The saved {@link com.flybits.commons.library.models.User#id} or "" if no
     * {@link com.flybits.commons.library.models.User#id} is saved.
     */
    public static String getUserID(Context context){
        SharedPreferences preferences   = getPreferences(context);
        return preferences.getString(FlyingConstants.PREF_USER_ID, FlyingConstants.PREF_DEFAULT_EMPTY_STRING);
    }

    /**
     * Indicates whether or not the user is logged in based on the previous network request.
     *
     * @param context The context of the application.
     * @return true if the previous request indicated that the application is logged in, false
     * otherwise.
     */
    public static boolean isFlybitsAuthenticated(Context context) {
        SharedPreferences preferences   = getPreferences(context);
        return preferences.getBoolean(FlyingConstants.PREF_AUTHENTICATION, false);
    }

    /**
     * Sets the unique device ID obtained from the Flybits Core.
     *
     * @param context The context of the application.
     * @param deviceID The unique device identifier that is used by Flybits to identify a device
     *                 using a GUID.
     */
    public static void setDeviceID(@NonNull Context context, @NonNull String deviceID) {
        setVariable(context, FlyingConstants.PREF_DEVICE_ID, deviceID);
    }

    /**
     * Sets whether the application is currently authenticated with the Flybits system. A user is
     * authenticated when they successfully call login.
     *
     * @param context The context of the application.
     * @param isAuthenticated The unique JWT Token that is used by Flybits to identify a user/device
     *                 combination.
     */
    public static void setFlybitsXAuthentication(Context context, boolean isAuthenticated) {
        setVariable(context, FlyingConstants.PREF_AUTHENTICATION, isAuthenticated);
    }

    /**
     * Sets the unique JWT Token obtained from the Flybits Core for the user/device combination.
     *
     * @param context The context of the application.
     * @param jwtToken The unique JWT Token that is used by Flybits to identify a user/device
     *                 combination.
     */
    public static void setJWTToken(Context context, @NonNull String jwtToken) {
        setVariable(context, FlyingConstants.PREF_JWT_TOKEN, jwtToken);
    }
    /**
     * Sets the localization values of the device.
     *
     * @param context The context of the application.
     * @param listOfLanguages The array of languages that should be used for this device.
     */
    public static void setLocalization(Context context, ArrayList<String> listOfLanguages) {
        SharedPreferences.Editor preferences = getPreferences(context).edit();
        preferences.putString(FlyingConstants.PREF_LANGUAGE_CODES, Utilities.convertLocalizationCodeToString(listOfLanguages));
        preferences.apply();
    }

    /**
     * Sets the unique identifier used to represent a unique user within the Flybits ecosystem.
     *
     * @param context The context of the application.
     * @param userID The unique user identifier used to uniquely identifier a user within the
     *               Flybits ecosystem.
     */
    public static void setUserID(Context context, @NonNull String userID) {
        setVariable(context, FlyingConstants.PREF_USER_ID, userID);
    }

    /**
     * Get the default shared preferences for the application.
     *
     * @param context The context of the application.
     * @return The default SharedPreferences object for the application.
     */
    public static SharedPreferences getPreferences(Context context){
        return context.getSharedPreferences("FLYBITS_PREF", MODE_PRIVATE );
    }

    private static void setVariable(Context context, @NonNull String key, @NonNull String value){
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putString(key, value);
        editor.apply();
    }

    private static void setVariable(Context context, @NonNull String key, boolean value){
        SharedPreferences.Editor editor = getPreferences(context).edit();
        editor.putBoolean(key, value);
        editor.apply();
    }
}
