package com.flybits.internal.db.models;

import androidx.room.ColumnInfo;
import androidx.room.Entity;
import androidx.room.Ignore;
import android.os.Parcel;
import android.os.Parcelable;
import androidx.annotation.NonNull;

import java.util.Objects;

/**
 * This class is the base model used for storing preferences within a local DB so that they can be
 * retrieved at any time regardless of whether there is Internet connectivity or not.
 */
@Entity(tableName = "preferences", primaryKeys = {"prefKey","value"})
public class Preference implements Parcelable{

    @ColumnInfo(name = "prefKey")
    @NonNull
    private String prefKey;

    @ColumnInfo(name = "value")
    @NonNull
    private String value;

    /**
     * Constructor used for un-flattening a {@code Preference} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Preference} parcel.
     */
    @Ignore
    public Preference(Parcel in){
        prefKey = Objects.requireNonNull(in.readString());
        value   = Objects.requireNonNull(in.readString());
    }

    /**
     * The constructor used to define all the attributes within the {@code Preference} object.
     *
     * @param prefKey The key under which the {@code Preference} is saved. You can multiple values
     *                saved under each key.
     * @param value The value that is saved under a specific {@code prefKey}.
     */
    public Preference(@NonNull String prefKey, @NonNull String value){
        this.prefKey    = prefKey;
        this.value      = value;
    }

    /**
     * Get the Preference Key under which the {@code value} will be saved under.
     *
     * @return The preference key under which the {@code value} will be saved under.
     */
    @NonNull
    public String getPrefKey() {
        return prefKey;
    }

    /**
     * Set the Preference Key under which the {@code value} will be saved under.
     *
     * @param prefKey The preference key under which the {@code value} will be saved under.
     */
    public void setPrefKey(@NonNull String prefKey) {
        this.prefKey = prefKey;
    }

    /**
     * Get the value of the preference being saved.
     *
     * @return The value of preference that is being saved.
     */
    @NonNull
    public String getValue() {
        return value;
    }

    /**
     * Set the value of the preference being saved.
     *
     * @param value The value of preference that is being saved.
     */
    public void setValue(@NonNull String value) {
        this.value = value;
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled
     * representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    public int describeContents() {
        return 0;
    }

    /**
     * Flatten this {@code Preference} into a Parcel.
     *
     * @param out The Parcel in which the {@code Preference} object should be written.
     * @param flags Additional flags about how the {@code Preference} object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(prefKey);
        out.writeString(value);
    }

    /**
     * Parcelable.Creator that instantiates {@code Preference} objects
     */
    public static final Creator<Preference> CREATOR = new Creator<Preference>(){
        public Preference createFromParcel(Parcel in){
            return new Preference(in);
        }

        public Preference[] newArray(int size){
            return new Preference[size];
        }
    };
}
