package com.flybits.commons.library.logging

/**
 * Logger is an internal class that used to log information to the logcat.
 */
object Logger {

    @JvmStatic
    internal val displayer = Displayer()

    /**
     * @return The [Displayer] that displays the information in the Logcat and sets the provided filter as a tag in it.
     *
     * @param filter The [String] representing filter that is set as a tag in [Displayer].
     */
    @JvmStatic
    fun appendTag(filter: String): Displayer {
        return displayer.appendTag(filter)
    }

    /**
     * Sets the [VerbosityLevel] for logging all messages with it within the SDKs.
     *
     * @param level The level of verbosity that should be displayed. Log levels enum values from [VerbosityLevel].
     * The default is level [VerbosityLevel.NONE].
     */
    @JvmStatic
    fun setVerbosity(level : VerbosityLevel) : Displayer{
        return displayer.setVerbosity(level)
    }

    /**
     * Creates and sends a debug log with provided [msg] to be displayed in the Logcat with [Displayer.DEFAULT_TAG]
     * based on current [VerbosityLevel].
     *
     * @param msg - The message to be displayed in the Logcat.
     */
    @JvmStatic
    fun d(msg: String) {
        displayer.appendTag(Displayer.DEFAULT_TAG).d(msg)
    }

    /**
     * Creates and sends an error log with provided [msg] to be displayed in the Logcat with [Displayer.DEFAULT_TAG]
     * based on current [VerbosityLevel].
     *
     * @param msg - The message to be displayed in the Logcat.
     */
    @JvmStatic
    fun e(msg: String) {
        displayer.appendTag(Displayer.DEFAULT_TAG).e(msg)
    }

    /**
     * Creates and sends an information log with provided [msg] to be displayed in the Logcat with [Displayer.DEFAULT_TAG]
     * based on current [VerbosityLevel].
     *
     * @param msg - The message to be displayed in the Logcat.
     */
    @JvmStatic
    fun i(msg: String) {
        displayer.appendTag(Displayer.DEFAULT_TAG).i(msg)
    }

    /**
     * Creates and sends a warning log with provided [msg] to be displayed in the Logcat with [Displayer.DEFAULT_TAG]
     * based on current [VerbosityLevel].
     *
     * @param msg - The message to be displayed in the Logcat.
     */
    @JvmStatic
    fun w(msg: String) {
        displayer.appendTag(Displayer.DEFAULT_TAG).w(msg)
    }

    /**
     * Creates and sends a log with provided [sdkMethod] as a message and [e] as [Exception] to be displayed in the
     * Logcat with [Displayer.DEFAULT_TAG] based on current [VerbosityLevel]. Also prints [e]'s information to the
     * stack trace.
     *
     * @param sdkMethod The name of the method where the error was caused.
     * @param e - The exception to display information about.
     *
     */
    @JvmStatic
    fun exception(sdkMethod: String, e: Exception) {
        displayer.appendTag(Displayer.DEFAULT_TAG).exception(sdkMethod, e)
    }
}
