package com.flybits.commons.library.api.results

import android.os.Handler
import com.flybits.commons.library.api.results.callbacks.ListResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.models.internal.Result
import java.util.*
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors

/**
 * The [ListResult] class is used to query the server when a list of objects is expected to
 * be returned, a [onSuccess] method or onException with an exception as the reason for
 * the failure will be returned.
 *
 * @param callback The [ListResultCallback] that indicates whether or not the network
 *                 request was successful or not, if successful a list of objects will be returned.
 * @param handler The handler that allows methods to be run in the UI.
 * @param service The [ExecutorService] that is used to run the network request thread.
 */
class ListResult<T>(val callback : ListResultCallback<T>? = null,
                    val handler : Handler? = null,
                    service: ExecutorService = Executors.newSingleThreadExecutor()) : FlybitsResult(service){

    /**
     * Sets the [FlybitsException] that was thrown when requesting more items to be loaded.
     *
     * @param e The [FlybitsException] that caused the network to fail.
     * @return true if the error has been set, false otherwise
     */
    fun setFailed(e: FlybitsException): Boolean {
        return if (failed()) {
            if (callback != null) {
                handler?.post { callback.onException(e) } ?: callback.onException(e)
            }
            true
        } else false
    }

    /**
     * Sets the result of the network request as successful and returns the expected object. This
     * object can be null.
     *
     * @return true if the request was successful, false otherwise
     */
    fun setSuccess(result : ArrayList<T>): Boolean {
        return if (success()) {
            if (callback != null) {
                handler?.post { callback.onSuccess(result) } ?: callback.onSuccess(result)
            }
            true
        } else false
    }

    /**
     * Set the [Result] of the network request.
     *
     * @param result The [Result] of the network request.
     * @return true if the request was successful, false if the request failed
     */
    fun setResult(result: Result<ArrayList<T>>) : Boolean {
        return if (result.status == RequestStatus.COMPLETED) {
            setSuccess(result.response)
        } else setFailed(result.exception)
    }
}