package com.flybits.commons.library.api

import android.content.Context
import com.flybits.commons.library.SharedElementsFactory.get
import com.flybits.commons.library.api.ScopeOfficer.onOptedStateChange
import com.flybits.commons.library.deserializations.IDeserializer
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.exceptions.InvalidJsonFormatException
import com.flybits.commons.library.exceptions.NetworkResponseException
import com.flybits.commons.library.http.HttpDefaultClass
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.logging.Logger.appendTag
import com.flybits.commons.library.models.internal.PagedResponse
import com.flybits.commons.library.models.internal.QueryParameters
import com.flybits.commons.library.models.internal.Result
import okhttp3.MediaType
import okhttp3.RequestBody

object FlyAway {
    internal const val TAG = "Network"

    @JvmStatic
    operator fun <T> get(
            mContext: Context,
            urlEndpoint: String,
            params: QueryParameters?,
            deserializer: IDeserializer<*>,
            uniqueIdentifier: String
    ): Result<PagedResponse<T>> {
        return get(mContext, urlEndpoint, null, params, deserializer, uniqueIdentifier)
    }

    @Suppress("UNCHECKED_CAST")
    @JvmStatic
    operator fun <T> get(
            context: Context,
            urlEndpoint: String,
            headers: HashMap<String, String>?,
            params: QueryParameters?,
            deserializer: IDeserializer<*>,
            uniqueIdentifier: String
    ): Result<PagedResponse<T>> {
        var result: Result<PagedResponse<T>>
        return try {
            val httpBuilder = HttpDefaultClass.Builder(context, true, urlEndpoint, params)
            if (headers != null) {
                httpBuilder.addHeaders(headers)
            }
            result = httpBuilder.get().response as Result<PagedResponse<T>>
            val response: PagedResponse<T>?
            if (result.status == RequestStatus.COMPLETED) {
                response = deserializer.fromJson(result.responseAsString) as PagedResponse<T>
                if (response != null) {
                    result.setResponse(response)
                } else {
                    appendTag(TAG)
                            .e("There appears to be a parsing error with " + result.responseAsString)
                    result = Result(
                            InvalidJsonFormatException(result.responseAsString), result
                            .responseAsString
                    )
                }
            } else if (result.status == RequestStatus.OPTED_OUT) {
                appendTag(TAG).i("User appears to be opted-out")
                onOptedStateChange(context, false)
            }
            result
        } catch (e: Exception) {
            if (get(context).getGatewayURL().isEmpty()) {
                appendTag(TAG).e("Gateway url appears to be null")
            } else {
                appendTag(TAG).e(uniqueIdentifier, e)
            }
            e.printStackTrace()
            result = if (e is ClassCastException) {
                Result(FlybitsException("Invalid Deserializer for this request: " + deserializer.javaClass), "")
            } else {
                Result(NetworkResponseException(), "")
            }
            result
        }
    }

    @Suppress("UNCHECKED_CAST")
    @JvmStatic
    operator fun <T> get(
            context: Context,
            urlEndpoint: String,
            headers: HashMap<String, String>?,
            params: QueryParameters?,
            uniqueIdentifier: String
    ): Result<T> {
        var result: Result<T>
        return try {
            val httpBuilder = HttpDefaultClass.Builder(context, true, urlEndpoint, params)
            if (headers != null) {
                httpBuilder.addHeaders(headers)
            }
            result = httpBuilder.get().response as Result<T>
            if (result.status == RequestStatus.COMPLETED) {
                val response: T? = result.responseAsString as T
                if (response != null) {
                    result.setResponse(response)
                } else {
                    appendTag(TAG)
                            .e("There appears to be a parsing error with " + result.responseAsString)
                    result = Result(
                            InvalidJsonFormatException(result.responseAsString), result
                            .responseAsString
                    )
                }
            } else if (result.status == RequestStatus.OPTED_OUT) {
                appendTag(TAG).i("User appears to be opted-out")
                onOptedStateChange(context, false)
            }
            result
        } catch (e: Exception) {
            if (get(context).getGatewayURL().isEmpty()) {
                appendTag(TAG).e("Gateway url appears to be null")
            } else {
                appendTag(TAG).e(uniqueIdentifier, e)
            }
            result = if (e is ClassCastException) {
                Result(FlybitsException("Something went wrong for this request."), "")
            } else {
                Result(NetworkResponseException(), "")
            }
            result
        }
    }

    @Suppress("UNCHECKED_CAST")
    @JvmStatic
    operator fun <T> get(
            context: Context,
            urlEndpoint: String,
            headers: HashMap<String, String>?,
            queryParameters: Map<String, String>?,
            deserializer: IDeserializer<*>,
            uniqueIdentifier: String
    ): Result<ArrayList<T>> {
        var result: Result<ArrayList<T>>
        return try {
            val httpBuilder = HttpDefaultClass.Builder(context, true, urlEndpoint, queryParameters)
            if (headers != null) {
                httpBuilder.addHeaders(headers)
            }
            result = httpBuilder.get().response as Result<ArrayList<T>>
            if (result.status == RequestStatus.COMPLETED) {
                val response = deserializer.fromJson(result.responseAsString) as ArrayList<T>
                if (response != null) {
                    result.setResponse(response)
                } else {
                    appendTag(TAG)
                            .e("There appears to be a parsing error with " + result.responseAsString)
                    result = Result(
                            InvalidJsonFormatException(result.responseAsString), result
                            .responseAsString
                    )
                }
            } else if (result.status == RequestStatus.OPTED_OUT) {
                appendTag(TAG).i("User appears to be opted-out")
                onOptedStateChange(context, false)
            }
            result
        } catch (e: Exception) {
            if (get(context).getGatewayURL().isEmpty()) {
                appendTag(TAG).e("Gateway url appears to be null")
            } else {
                appendTag(TAG).e(uniqueIdentifier, e)
            }
            e.printStackTrace()
            result = if (e is ClassCastException) {
                Result(
                        FlybitsException("Invalid Deserializer for this request: " + deserializer.javaClass), ""
                )
            } else {
                Result(NetworkResponseException(), "")
            }
            result
        }
    }

    @JvmStatic
    operator fun <T> get(
            context: Context,
            urlEndpoint: String,
            deserializer: IDeserializer<*>?,
            uniqueIdentifier: String,
            type: Class<T>?
    ): Result<T> {
        return get(context, urlEndpoint, null, deserializer, uniqueIdentifier, type)
    }

    @Suppress("UNCHECKED_CAST")
    @JvmStatic
    operator fun <T> get(
            context: Context,
            urlEndpoint: String,
            headers: HashMap<String, String>?,
            deserializer: IDeserializer<*>?,
            uniqueIdentifier: String,
            type: Class<T>?
    ): Result<T> {
        var result: Result<T>
        return try {
            val httpBuilder = HttpDefaultClass.Builder(context, true, urlEndpoint)
            if (headers != null) {
                httpBuilder.addHeaders(headers)
            }
            result = httpBuilder.get().response as Result<T>

            if (result.status == RequestStatus.COMPLETED && deserializer != null) {
                val response = type?.cast(deserializer.fromJson(result.responseAsString))
                if (response != null) {
                    result.setResponse(response)
                } else {
                    appendTag(TAG)
                            .e("There appears to be a parsing error with " + result.responseAsString)
                    result = Result(
                            InvalidJsonFormatException(result.responseAsString), result
                            .responseAsString
                    )
                }
            } else if (result.status == RequestStatus.OPTED_OUT) {
                appendTag(TAG).i("User appears to be opted-out")
                onOptedStateChange(context, false)
            }
            result
        } catch (e: Exception) {
            if (get(context).getGatewayURL().isEmpty()) {
                appendTag(TAG).e("Gateway url appears to be null")
            } else {
                appendTag(TAG).e(uniqueIdentifier, e)
            }
            e.printStackTrace()
            result = if (e is ClassCastException) {
                Result(FlybitsException("Invalid Deserializer for this request: " + deserializer!!.javaClass), "")
            } else {
                Result(NetworkResponseException(), "")
            }
            result
        }
    }

    @JvmStatic
    fun <T> post(
            context: Context,
            urlEndpoint: String,
            json: String?,
            deserializer: IDeserializer<*>?,
            uniqueIdentifier: String,
            type: Class<T>?
    ): Result<T> {
        val requestBody =
                RequestBody.create(MediaType.parse("application/json; charset=utf-8"), json ?: "")
        return post(context, urlEndpoint, null, requestBody, deserializer, uniqueIdentifier, type)
    }

    @JvmStatic
    fun <T> post(
            context: Context,
            urlEndpoint: String,
            json: String?,
            deserializer: IDeserializer<*>?,
            uniqueIdentifier: String,
            type: Class<T>?,
            headers: HashMap<String, String>?
    ): Result<T> {
        val requestBody =
                RequestBody.create(MediaType.parse("application/json; charset=utf-8"), json ?: "")
        return post(
                context,
                urlEndpoint,
                headers,
                requestBody,
                deserializer,
                uniqueIdentifier,
                type
        )
    }

    @JvmStatic
    fun <T> post(
            context: Context,
            urlEndpoint: String,
            requestBody: RequestBody?,
            deserializer: IDeserializer<*>?,
            uniqueIdentifier: String,
            type: Class<T>?
    ): Result<T> {
        return post(
                context,
                urlEndpoint,
                null,
                requestBody ?: RequestBody.create(
                        MediaType.parse("application/json; charset=utf-8"),
                        ""
                ),
                deserializer,
                uniqueIdentifier,
                type
        )
    }

    @Suppress("UNCHECKED_CAST")
    @JvmStatic
    fun <T> post(
            context: Context,
            urlEndpoint: String,
            headers: HashMap<String, String>?,
            requestBody: RequestBody?,
            deserializer: IDeserializer<*>?,
            uniqueIdentifier: String,
            type: Class<T>?
    ): Result<T> {
        var result: Result<T>
        return try {
            val httpBuilder = HttpDefaultClass.Builder(context, true, urlEndpoint)
            if (headers != null) {
                httpBuilder.addHeaders(headers)
            }
            result = httpBuilder.post(requestBody).response as Result<T>
            if (result.status == RequestStatus.COMPLETED && deserializer != null) {
                val response = type?.cast(deserializer.fromJson(result.responseAsString))
                if (response != null) {
                    result.setResponse(response)
                } else {
                    appendTag(TAG)
                            .e("There appears to be a parsing error with " + result.responseAsString)
                    result = Result(
                            InvalidJsonFormatException(result.responseAsString), result
                            .responseAsString
                    )
                }
            } else if (result.status == RequestStatus.OPTED_OUT) {
                appendTag(TAG).i("User appears to be opted-out")
                onOptedStateChange(context, false)
            }
            result
        } catch (e: Exception) {
            if (get(context).getGatewayURL().isEmpty()) {
                appendTag(TAG).e("Gateway url appears to be null")
            } else {
                appendTag(TAG).e(uniqueIdentifier, e)
            }
            e.printStackTrace()
            result = if (e is ClassCastException) {
                Result(FlybitsException("Invalid Deserializer for this request: " + deserializer!!.javaClass), "")
            } else {
                Result(NetworkResponseException(), "")
            }
            result
        }
    }

    @JvmStatic
    fun <T> put(
            context: Context,
            urlEndpoint: String,
            requestBody: RequestBody,
            deserializer: IDeserializer<*>?,
            uniqueIdentifier: String,
            type: Class<T>?
    ): Result<T> {
        return put(context, urlEndpoint, null, requestBody, deserializer, uniqueIdentifier, type)
    }

    @JvmStatic
    fun <T> put(
            context: Context,
            urlEndpoint: String,
            json: String,
            deserializer: IDeserializer<*>?,
            uniqueIdentifier: String,
            type: Class<T>?
    ): Result<T> {
        val requestBody = RequestBody.create(
                MediaType.parse("application/json; charset=utf-8"),
                json
        )
        return put(context, urlEndpoint, null, requestBody, deserializer, uniqueIdentifier, type)
    }

    @JvmStatic
    fun <T> put(
            context: Context,
            urlEndpoint: String,
            json: String,
            deserializer: IDeserializer<*>?,
            uniqueIdentifier: String,
            type: Class<T>?,
            headers: HashMap<String, String>?
    ): Result<T> {
        val requestBody = RequestBody.create(
                MediaType.parse("application/json; charset=utf-8"),
                json
        )
        return put(
                context,
                urlEndpoint,
                headers,
                requestBody,
                deserializer,
                uniqueIdentifier,
                type
        )
    }

    @Suppress("UNCHECKED_CAST")
    @JvmStatic
    fun <T> put(
            context: Context,
            urlEndpoint: String,
            headers: HashMap<String, String>?,
            requestBody: RequestBody,
            deserializer: IDeserializer<*>?,
            uniqueIdentifier: String,
            type: Class<T>?
    ): Result<T> {
        var result: Result<T>
        return try {
            val httpBuilder = HttpDefaultClass.Builder(context, true, urlEndpoint)
            if (headers != null) {
                httpBuilder.addHeaders(headers)
            }
            result = httpBuilder.put(requestBody).response as Result<T>
            if (result.status == RequestStatus.COMPLETED && deserializer != null) {
                val response = type?.cast(deserializer.fromJson(result.responseAsString))
                if (response != null) {
                    result.setResponse(response)
                } else {
                    appendTag(TAG)
                            .e("There appears to be a parsing error with " + result.responseAsString)
                    result = Result(
                            InvalidJsonFormatException(result.responseAsString), result.responseAsString
                    )
                }
            } else if (result.status == RequestStatus.OPTED_OUT) {
                appendTag(TAG).i("User appears to be opted-out")
                onOptedStateChange(context, false)
            }
            result
        } catch (e: Exception) {
            if (get(context).getGatewayURL().isEmpty()) {
                appendTag(TAG).e("Gateway url appears to be null")
            } else {
                appendTag(TAG).e(uniqueIdentifier, e)
            }
            e.printStackTrace()
            result = if (e is ClassCastException) {
                Result(
                        FlybitsException("Invalid Deserializer for this request: " + deserializer!!.javaClass), ""
                )
            } else {
                Result(NetworkResponseException(), "")
            }
            result
        }
    }

    @JvmStatic
    fun delete(
            context: Context,
            urlEndpoint: String,
            uniqueIdentifier: String,
            id: String?
    ): Result<*> {
        return delete(context, urlEndpoint, null, uniqueIdentifier, id)
    }

    @JvmStatic
    fun delete(
            context: Context,
            urlEndpoint: String,
            headers: HashMap<String, String>?,
            uniqueIdentifier: String,
            id: String?
    ): Result<Any> {
        val validatedUrlEndpoint = if (id != null) {
            "$urlEndpoint/$id"
        } else {
            urlEndpoint
        }
        var result: Result<Any>
        return try {
            val httpBuilder = HttpDefaultClass.Builder(context, true, validatedUrlEndpoint)
            if (headers != null) {
                httpBuilder.addHeaders(headers)
            }
            result = httpBuilder.delete().response
            if (result.status == RequestStatus.OPTED_OUT) {
                appendTag(TAG).i("User appears to be opted-out")
                onOptedStateChange(context, false)
            }
            result
        } catch (e: Exception) {
            if (get(context).getGatewayURL().isEmpty()) {
                appendTag(TAG).e("Gateway url appears to be null")
            } else {
                appendTag(TAG).e(uniqueIdentifier, e)
            }
            e.printStackTrace()
            result = Result(NetworkResponseException(), "")
            result
        }
    }
}