package com.flybits.commons.library.models

import android.content.Context
import android.os.Handler
import android.os.Looper
import com.flybits.commons.library.api.FlyAway
import com.flybits.commons.library.api.results.ObjectResult
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.exceptions.InvalidJsonFormatException
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.models.internal.Result
import org.json.JSONException
import org.json.JSONObject
import java.util.concurrent.Executors

/**
 * Class to hold and provide data about ES project settings and configuration.
 */
class Project(

        /**
         * @return The id of this project.
         */
        val id: String,

        /**
         * @return The id of the owner that created this project.
         */
        val creatorId: String,

        /**
         * @return The name of this project.
         */
        val name: String,

        /**
         * @return The subdomain.
         */
        val subdomain: String,

        /**
         * @return The tier level of this project.
         */
        val tierId: String,

        /**
         * @return True if project is active.
         */
        val isActive: Boolean) {

    companion object {
        private const val URL_SETTINGS = "/sso/settings/configs"
        private const val TYPE_PROJECT_SETTINGS = "project-settings"

        /**
         * @return The [ObjectResult] which contains information about network request, gets all the settings for a
         * Project.
         *
         * @param context  The context of the activity that is calling this method.
         * @param clazz    The Settings class that implements [JsonParser] so that Strings can be
         * serialized and deserialized more easily.
         * @param callback Indicates where or not the network request was successful.
         * @param <T>      The class that implements [JsonParser].
         * @param handler  The handler that the callback will be invoked through.
         */
        fun <T : JsonParser?> getProjectSettings(context: Context?, clazz: T,
                callback: ObjectResultCallback<T>?, handler: Handler): ObjectResult<T> {
            return getSettings(TYPE_PROJECT_SETTINGS, context, clazz, callback, handler)
        }

        /**
         * @return The [ObjectResult] which contains information about network request, gets all the settings for a
         * Project.
         *
         * @param context  The context of the activity that is calling this method.
         * @param clazz    The Settings class that implements [JsonParser] so that Strings can be
         * serialized and deserialized more easily.
         * @param callback Indicates where or not the network request was successful.
         * @param <T>      The class that implements [JsonParser].
         */
        fun <T : JsonParser?> getProjectSettings(context: Context?, clazz: T,
                callback: ObjectResultCallback<T>?): ObjectResult<T> {
            return getSettings(TYPE_PROJECT_SETTINGS, context, clazz, callback, Handler(Looper.getMainLooper()))
        }

        /**
         * @return The [ObjectResult] which contains information about network request,gets all the settings of a
         * specific type.
         *
         * @param type     The type of settings being retrieved, or more specifically the {this} in: /sso/settings/configs/{this}
         * @param context  The context of the activity that is calling this method.
         * @param clazz    The Settings class that implements [JsonParser] so that Strings can be
         * serialized and deserialized more easily.
         * @param callback Indicates where or not the network request was successful.
         * @param <T>      The class that implements [JsonParser].
         */
        fun <T : JsonParser?> getSettings(type: String,
                context: Context?, clazz: T, callback: ObjectResultCallback<T>?): ObjectResult<T> {
            return getSettings(type, context, clazz, callback, Handler(Looper.getMainLooper()))
        }

        /**
         * @return The [ObjectResult] which contains information about network request, gets all the settings of a
         * specific type.
         *
         * @param type     The type of settings being retrieved, or more specifically the {this} in: /sso/settings/configs/{this}
         * @param context  The context of the activity that is calling this method.
         * @param clazz    The Settings class that implements [JsonParser] so that Strings can be
         * serialized and deserialized more easily.
         * @param callback Indicates where or not the network request was successful.
         * @param <T>      The class that implements [JsonParser].
         * @param handler  The handler that the callback will be invoked through.
         */
        fun <T : JsonParser?> getSettings(type: String,
                context: Context?, clazz: T, callback: ObjectResultCallback<T>?, handler: Handler): ObjectResult<T> {
            val executorService = Executors.newSingleThreadExecutor()
            val request = ObjectResult(callback, handler, executorService)
            executorService.execute {
                try {
                    val url = String.format("%s/%s", URL_SETTINGS, type)
                    val getSettings: Result<T> = FlyAway.get(context!!, url, null, "Project.GetSettings", null)
                    //Check to make sure that there is properties that should be returned
                    if (getSettings.status == RequestStatus.COMPLETED) {
                        //Parse the returned object based on the fromJSON(...) method
                        val response = getSettings.responseAsString
                        val `object` = JSONObject(response)
                        if (!`object`.isNull("settings")) {
                            val settingsObject = `object`.getJSONObject("settings")
                            clazz!!.fromJson(settingsObject.toString())
                            request.setSuccess(clazz)
                        } else {
                            request.setFailed(InvalidJsonFormatException())
                        }
                    } else if (getSettings.status == RequestStatus.NOT_FOUND) {
                        request.setSuccess(clazz)
                    } else {
                        request.setResult(getSettings)
                    }
                } catch (e: FlybitsException) {
                    request.setFailed(e)
                } catch (e: ClassCastException) {
                    val exception = FlybitsException(e)
                    request.setFailed(exception)
                } catch (e: JSONException) {
                    val exception = FlybitsException(e)
                    request.setFailed(exception)
                }
            }
            return request
        }
    }
}