package com.flybits.commons.library.api

import android.content.Context
import com.flybits.commons.library.models.User

/**
 * The `FlybitsScope` class is an abstract class that is used to define an SDK scope which is
 * used to define the properties of an SDK. For example, the Flybits Context SDK would extend this
 * `FlybitsScope` class which will then be used when defining the
 * [com.flybits.commons.library.api.FlybitsManager.Builder.addScope].
 */
abstract class FlybitsScope
/**
 * This constructor is used to define this [FlybitsScope] based on a unique Scope code.
 *
 * @param name The unique name of the Scope.
 */(
        /**
         * Get the unique name of the Scope.
         *
         * @return The unique name of the Scope.
         */
        val scopeName: String) {

    /**
     * Indicates that the login was successfully made and that the appropriate actions for an SDK
     * should be performed.
     *
     * @param context The current Context of the application
     * @param user    The logged in [User].
     */
    abstract fun onConnected(context: Context, user: User)

    /**
     * Indicates that the specific scope is added. The SDK will need to make the appropriate actions.
     * @param authState Refers to status in [ScopeAuthState]
     */
    abstract fun onStart(authState: ScopeAuthState)

    /**
     * Indicates that the specific scope is removed. The SDK will need to make the appropriate actions.
     */
    abstract fun onStop()

    /**
     * Indicates that the logout was successful. The SDK will need to make the appropriate actions.
     *
     * @param context  The current Context of the application.
     */
    abstract fun onDisconnected(context: Context)

    /**
     * Indicates the logged in [User] has changed the opted state.
     *
     * @param context    The current Context of the application
     * @param optedState True if the user opted in, false if the user opted out.
     */
    abstract fun onOptedStateChange(context: Context, optedState: Boolean)

    /**
     * Indicates the authentication flow has not been successful and isConnected will return
     * false plus all content requests or context uploads will fail with 401.
     */
    abstract fun onFailedToConnect()

    abstract fun onConnecting()
}

enum class ScopeAuthState {
    DISCONNECTED,
    CONNECTED_OPT_IN,
    CONNECTED_OPT_OUT,
    CONNECTING,
    INITIATED
}