/*
 * Decompiled with CFR 0.152.
 */
package com.feedzai.commons.sql.abstraction.batch;

import com.feedzai.commons.sql.abstraction.FailureListener;
import com.feedzai.commons.sql.abstraction.batch.BatchEntry;
import com.feedzai.commons.sql.abstraction.engine.DatabaseEngine;
import com.feedzai.commons.sql.abstraction.engine.DatabaseEngineException;
import com.feedzai.commons.sql.abstraction.entry.EntityEntry;
import com.feedzai.commons.sql.abstraction.listeners.BatchListener;
import com.google.common.base.Strings;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nullable;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;

public abstract class AbstractBatch
implements Runnable {
    protected final Logger logger = LoggerFactory.getLogger(AbstractBatch.class);
    public static final int NO_RETRY = 0;
    public static final long DEFAULT_RETRY_INTERVAL = 300L;
    protected static final Marker dev = MarkerFactory.getMarker((String)"DEV");
    protected static final int salt = 100;
    private final Lock bufferLock = new ReentrantLock();
    private final Lock flushTransactionLock = new ReentrantLock();
    protected final DatabaseEngine de;
    protected final long maxAwaitTimeShutdown;
    protected ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1);
    protected final int batchSize;
    protected final long batchTimeout;
    protected int batch;
    protected volatile long lastFlush;
    protected LinkedList<BatchEntry> buffer = new LinkedList();
    protected String name;
    protected Optional<BatchListener> batchListener = Optional.empty();
    protected final int maxFlushRetries;
    protected final long flushRetryDelay;

    protected AbstractBatch(DatabaseEngine de, String name, int batchSize, long batchTimeout, long maxAwaitTimeShutdown, @Nullable BatchListener batchListener, int maxFlushRetries, long flushRetryDelay) {
        Objects.requireNonNull(de, "The provided database engine is null.");
        this.de = de;
        this.batchSize = batchSize;
        this.batch = batchSize;
        this.batchTimeout = batchTimeout;
        this.lastFlush = System.currentTimeMillis();
        this.name = Strings.isNullOrEmpty((String)name) ? "Anonymous Batch" : name;
        this.maxAwaitTimeShutdown = maxAwaitTimeShutdown;
        this.batchListener = Optional.ofNullable(batchListener);
        this.maxFlushRetries = maxFlushRetries;
        this.flushRetryDelay = flushRetryDelay;
    }

    protected AbstractBatch(DatabaseEngine de, String name, int batchSize, long batchTimeout, long maxAwaitTimeShutdown, @Nullable BatchListener batchListener) {
        this(de, name, batchSize, batchTimeout, maxAwaitTimeShutdown, batchListener, 0, 300L);
    }

    @Deprecated
    protected AbstractBatch(DatabaseEngine de, String name, int batchSize, long batchTimeout, long maxAwaitTimeShutdown, FailureListener failureListener, int maxFlushRetries, long flushRetryDelay) {
        this(de, name, batchSize, batchTimeout, maxAwaitTimeShutdown, AbstractBatch.convertToBatchListener(failureListener), maxFlushRetries, flushRetryDelay);
    }

    @Deprecated
    protected AbstractBatch(DatabaseEngine de, String name, int batchSize, long batchTimeout, long maxAwaitTimeShutdown, FailureListener failureListener) {
        this(de, name, batchSize, batchTimeout, maxAwaitTimeShutdown, AbstractBatch.convertToBatchListener(failureListener), 0, 300L);
    }

    protected AbstractBatch(DatabaseEngine de, String name, int batchSize, long batchTimeout, long maxAwaitTimeShutdown) {
        this(de, name, batchSize, batchTimeout, maxAwaitTimeShutdown, (BatchListener)null);
    }

    protected AbstractBatch(DatabaseEngine de, int batchSize, long batchTimeout, long maxAwaitTimeShutdown) {
        this(de, null, batchSize, batchTimeout, maxAwaitTimeShutdown);
    }

    protected void start() {
        Runnable resilientTask = () -> {
            try {
                this.run();
            }
            catch (Exception e) {
                this.logger.error("[{}] Error during timeout-initiated flush", (Object)this.name, (Object)e);
            }
        };
        this.scheduler.scheduleAtFixedRate(resilientTask, 0L, this.batchTimeout + 100L, TimeUnit.MILLISECONDS);
    }

    public void destroy() {
        this.logger.trace("{} - Destroy called on Batch", (Object)this.name);
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(this.maxAwaitTimeShutdown, TimeUnit.MILLISECONDS)) {
                this.logger.warn("Could not terminate batch within {}. Forcing shutdown.", (Object)DurationFormatUtils.formatDurationWords((long)this.maxAwaitTimeShutdown, (boolean)true, (boolean)true));
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.logger.debug("Interrupted while waiting.", (Throwable)e);
        }
        this.flush();
    }

    public void add(BatchEntry batchEntry) throws DatabaseEngineException {
        this.bufferLock.lock();
        try {
            this.buffer.add(batchEntry);
            --this.batch;
        }
        finally {
            this.bufferLock.unlock();
        }
        if (this.batch <= 0) {
            this.flush();
        }
    }

    public void add(String entityName, EntityEntry ee) throws DatabaseEngineException {
        this.add(new BatchEntry(entityName, ee));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void flush() {
        LinkedList<BatchEntry> temp;
        this.bufferLock.lock();
        try {
            this.lastFlush = System.currentTimeMillis();
            if (this.batch == this.batchSize) {
                this.logger.trace("[{}] Batch empty, not flushing", (Object)this.name);
                return;
            }
            this.batch = this.batchSize;
            temp = this.buffer;
            this.buffer = new LinkedList();
        }
        finally {
            this.bufferLock.unlock();
        }
        long start = System.currentTimeMillis();
        try {
            this.flushTransactionLock.lock();
            start = System.currentTimeMillis();
            this.processBatch(temp);
            this.onFlushSuccess(temp.toArray(new BatchEntry[0]));
            this.logger.trace("[{}] Batch flushed. Took {} ms, {} rows.", new Object[]{this.name, System.currentTimeMillis() - start, temp.size()});
            return;
        }
        catch (Exception e) {
            int retryCount;
            if (this.maxFlushRetries > 0) {
                this.logger.warn(dev, "[{}] Error occurred while flushing. Retrying.", (Object)this.name, (Object)e);
            }
            boolean success = false;
            for (retryCount = 0; retryCount < this.maxFlushRetries && !success; ++retryCount) {
                try {
                    Thread.sleep(this.flushRetryDelay);
                    if (this.de.checkConnection() && this.de.isTransactionActive()) {
                        this.de.rollback();
                    }
                    this.processBatch(temp);
                    success = true;
                    continue;
                }
                catch (InterruptedException ex) {
                    this.logger.debug("Interrupted while trying to flush batch. Stopping retries.");
                    Thread.currentThread().interrupt();
                    break;
                }
                catch (Exception ex) {
                    this.logger.warn(dev, "[{}] Error occurred while flushing (retry attempt {}).", new Object[]{this.name, retryCount + 1, ex});
                }
            }
            if (success) {
                this.onFlushSuccess(temp.toArray(new BatchEntry[0]));
                this.logger.trace("[{}] Batch flushed. Took {} ms, {} retries, {} rows.", new Object[]{this.name, System.currentTimeMillis() - start, retryCount, temp.size()});
                return;
            }
            try {
                if (this.de.isTransactionActive()) {
                    this.de.rollback();
                }
            }
            catch (Exception ee) {
                ee.addSuppressed(e);
                this.logger.trace("[{}] Batch failed to check the flush transaction state", (Object)this.name, (Object)ee);
            }
            this.onFlushFailure(temp.toArray(new BatchEntry[0]));
            this.logger.error(dev, "[{}] Error occurred while flushing. Aborting batch flush.", (Object)this.name, (Object)e);
            return;
        }
        finally {
            try {
                if (this.de.isTransactionActive()) {
                    this.de.rollback();
                }
            }
            catch (Exception e) {
                this.logger.trace("[{}] Batch failed to check the flush transaction state", (Object)this.name, (Object)e);
            }
            finally {
                this.flushTransactionLock.unlock();
            }
        }
    }

    public void flush(boolean sync) {
        if (!sync) {
            this.flush();
        } else {
            try {
                this.flushTransactionLock.lock();
                this.flush();
            }
            finally {
                this.flushTransactionLock.unlock();
            }
        }
    }

    public void onFlushFailure(BatchEntry[] entries) {
        this.batchListener.ifPresent(batchListener -> batchListener.onFailure(entries));
    }

    public void onFlushSuccess(BatchEntry[] entries) {
        this.batchListener.ifPresent(batchListener -> batchListener.onSuccess(entries));
    }

    @Override
    public void run() {
        if (System.currentTimeMillis() - this.lastFlush >= this.batchTimeout) {
            this.logger.trace("[{}] Flush timeout occurred", (Object)this.name);
            this.flush();
        }
    }

    private void processBatch(List<BatchEntry> batchEntries) throws DatabaseEngineException {
        this.de.beginTransaction();
        for (BatchEntry entry : batchEntries) {
            this.de.addBatch(entry.getTableName(), entry.getEntityEntry());
        }
        this.de.flush();
        this.de.commit();
    }

    public static BatchListener convertToBatchListener(final FailureListener failureListener) {
        return new BatchListener(){

            @Override
            public void onFailure(BatchEntry[] rowsFailed) {
                failureListener.onFailure(rowsFailed);
            }

            @Override
            public void onSuccess(BatchEntry[] rowsSucceeded) {
            }
        };
    }
}

