package com.fasterxml.jackson.module.scala

import com.fasterxml.jackson.databind.Module.SetupContext
import com.fasterxml.jackson.module.scala.deser.{ScalaNumberDeserializersModule, ScalaObjectDeserializerModule, UntypedObjectDeserializerModule}
import com.fasterxml.jackson.module.scala.introspect.ScalaAnnotationIntrospectorModule

/**
 * Build a customised Scala Module.
 * @since 2.19.0
 */
object ScalaModule {

  class Builder {
    private val modules = scala.collection.mutable.Buffer[JacksonModule]()

    def addModule(module: JacksonModule): Builder = {
      modules.+=(module)
      this
    }

    def removeModule(module: JacksonModule): Builder = {
      modules.-=(module)
      this
    }

    def hasModule(module: JacksonModule): Boolean = {
      modules.contains(module)
    }

    def addAllBuiltinModules(): Builder = {
      addModule(IteratorModule)
      addModule(EnumerationModule)
      addModule(OptionModule)
      addModule(SeqModule)
      addModule(TupleModule)
      addModule(MapModule)
      addModule(SetModule)
      addModule(ScalaNumberDeserializersModule)
      addModule(ScalaAnnotationIntrospectorModule)
      addModule(ScalaObjectDeserializerModule)
      addModule(UntypedObjectDeserializerModule)
      addModule(EitherModule)
      addModule(SymbolModule)
      BuiltinModules.addScalaVersionSpecificModules(this)
      this
    }

    def build(): JacksonModule = new JacksonModule {
      override def getModuleName: String = "ScalaModule"
      override def setupModule(context: SetupContext): Unit = {
        modules.foreach(_.setupModule(context))
      }
    }
  }

  private var deserializeNullCollectionsAsEmpty = true

  def builder(): Builder = new Builder()

  /**
   * @return whether the module should support deserializing null collections as empty (default: true)
   */
  def shouldDeserializeNullCollectionsAsEmpty(): Boolean = deserializeNullCollectionsAsEmpty

  def deserializeNullCollectionsAsEmpty(asEmpty: Boolean): Unit = {
    deserializeNullCollectionsAsEmpty = asEmpty
  }
}

