/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.decoder.raw;

import com.facebook.presto.decoder.DecoderColumnHandle;
import com.facebook.presto.decoder.DecoderErrorCode;
import com.facebook.presto.decoder.FieldValueProvider;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.type.BigintType;
import com.facebook.presto.spi.type.BooleanType;
import com.facebook.presto.spi.type.DoubleType;
import com.facebook.presto.spi.type.IntegerType;
import com.facebook.presto.spi.type.SmallintType;
import com.facebook.presto.spi.type.TinyintType;
import com.facebook.presto.spi.type.Type;
import com.facebook.presto.spi.type.Varchars;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class RawColumnDecoder {
    private static final Pattern MAPPING_PATTERN = Pattern.compile("(\\d+)(?::(\\d+))?");
    private final String columnName;
    private final Type columnType;
    private final FieldType fieldType;
    private final int start;
    private final OptionalInt end;

    public RawColumnDecoder(DecoderColumnHandle columnHandle) {
        try {
            Objects.requireNonNull(columnHandle, "columnHandle is null");
            Preconditions.checkArgument((!columnHandle.isInternal() ? 1 : 0) != 0, (String)"unexpected internal column '%s'", (Object)columnHandle.getName());
            Preconditions.checkArgument((columnHandle.getFormatHint() == null ? 1 : 0) != 0, (String)"unexpected format hint '%s' defined for column '%s'", (Object)columnHandle.getFormatHint(), (Object)columnHandle.getName());
            this.columnName = columnHandle.getName();
            this.columnType = columnHandle.getType();
            try {
                this.fieldType = columnHandle.getDataFormat() == null ? FieldType.BYTE : FieldType.valueOf(columnHandle.getDataFormat().toUpperCase(Locale.ENGLISH));
            }
            catch (IllegalArgumentException e) {
                throw new IllegalArgumentException(String.format("invalid dataFormat '%s' for column '%s'", columnHandle.getDataFormat(), this.columnName));
            }
            String mapping = Optional.ofNullable(columnHandle.getMapping()).orElse("0");
            Matcher mappingMatcher = MAPPING_PATTERN.matcher(mapping);
            if (!mappingMatcher.matches()) {
                throw new IllegalArgumentException(String.format("invalid mapping format '%s' for column '%s'", mapping, this.columnName));
            }
            this.start = Integer.parseInt(mappingMatcher.group(1));
            this.end = mappingMatcher.group(2) != null ? OptionalInt.of(Integer.parseInt(mappingMatcher.group(2))) : (!Varchars.isVarcharType((Type)this.columnType) ? OptionalInt.of(this.start + this.fieldType.getSize()) : OptionalInt.empty());
            Preconditions.checkArgument((this.start >= 0 ? 1 : 0) != 0, (String)"start offset %s for column '%s' must be greater or equal 0", (int)this.start, (Object)this.columnName);
            this.end.ifPresent(endValue -> {
                Preconditions.checkArgument((endValue >= 0 ? 1 : 0) != 0, (String)"end offset %s for column '%s' must be greater or equal 0", (int)endValue, (Object)this.columnName);
                Preconditions.checkArgument((endValue >= this.start ? 1 : 0) != 0, (String)"end offset %s for column '%s' must greater or equal start offset", (int)endValue, (Object)this.columnName);
            });
            Preconditions.checkArgument((boolean)RawColumnDecoder.isSupportedType(this.columnType), (String)"Unsupported column type '%s' for column '%s'", (Object)this.columnType.getDisplayName(), (Object)this.columnName);
            if (this.columnType == BigintType.BIGINT) {
                this.checkFieldTypeOneOf(this.fieldType, this.columnName, FieldType.BYTE, FieldType.SHORT, FieldType.INT, FieldType.LONG);
            }
            if (this.columnType == IntegerType.INTEGER) {
                this.checkFieldTypeOneOf(this.fieldType, this.columnName, FieldType.BYTE, FieldType.SHORT, FieldType.INT);
            }
            if (this.columnType == SmallintType.SMALLINT) {
                this.checkFieldTypeOneOf(this.fieldType, this.columnName, FieldType.BYTE, FieldType.SHORT);
            }
            if (this.columnType == TinyintType.TINYINT) {
                this.checkFieldTypeOneOf(this.fieldType, this.columnName, FieldType.BYTE);
            }
            if (this.columnType == BooleanType.BOOLEAN) {
                this.checkFieldTypeOneOf(this.fieldType, this.columnName, FieldType.BYTE, FieldType.SHORT, FieldType.INT, FieldType.LONG);
            }
            if (this.columnType == DoubleType.DOUBLE) {
                this.checkFieldTypeOneOf(this.fieldType, this.columnName, FieldType.DOUBLE, FieldType.FLOAT);
            }
            if (Varchars.isVarcharType((Type)this.columnType)) {
                this.checkFieldTypeOneOf(this.fieldType, this.columnName, FieldType.BYTE);
            }
            if (!Varchars.isVarcharType((Type)this.columnType)) {
                Preconditions.checkArgument((!this.end.isPresent() || this.end.getAsInt() - this.start == this.fieldType.getSize() ? 1 : 0) != 0, (String)"Bytes mapping for column '%s' does not match dataFormat '%s'; expected %s bytes but got %s", (Object)this.columnName, (Object)this.fieldType.getSize(), (Object)(this.end.getAsInt() - this.start));
            }
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_USER_ERROR, (Throwable)e);
        }
    }

    private static boolean isSupportedType(Type type) {
        if (Varchars.isVarcharType((Type)type)) {
            return true;
        }
        return ImmutableList.of((Object)BigintType.BIGINT, (Object)IntegerType.INTEGER, (Object)SmallintType.SMALLINT, (Object)TinyintType.TINYINT, (Object)BooleanType.BOOLEAN, (Object)DoubleType.DOUBLE).contains((Object)type);
    }

    private void checkFieldTypeOneOf(FieldType declaredFieldType, String columnName, FieldType ... allowedFieldTypes) {
        if (!Arrays.asList(allowedFieldTypes).contains((Object)declaredFieldType)) {
            throw new IllegalArgumentException(String.format("Wrong dataFormat '%s' specified for column '%s'; %s type implies use of %s", declaredFieldType.name(), columnName, this.columnType.getDisplayName(), Joiner.on((String)"/").join((Object[])allowedFieldTypes)));
        }
    }

    public FieldValueProvider decodeField(byte[] value) {
        Objects.requireNonNull(value, "value is null");
        int actualEnd = this.end.orElse(value.length);
        if (this.start > value.length) {
            throw new PrestoException((ErrorCodeSupplier)DecoderErrorCode.DECODER_CONVERSION_NOT_SUPPORTED, String.format("start offset %s for column '%s' must be less that or equal to value length %s", this.start, this.columnName, value.length));
        }
        if (actualEnd > value.length) {
            throw new PrestoException((ErrorCodeSupplier)DecoderErrorCode.DECODER_CONVERSION_NOT_SUPPORTED, String.format("end offset %s for column '%s' must be less that or equal to value length %s", actualEnd, this.columnName, value.length));
        }
        return new RawValueProvider(ByteBuffer.wrap(value, this.start, actualEnd - this.start), this.fieldType, this.columnName, this.columnType);
    }

    private static class RawValueProvider
    extends FieldValueProvider {
        private final ByteBuffer value;
        private final FieldType fieldType;
        private final String columnName;
        private final Type columnType;
        private final int size;

        public RawValueProvider(ByteBuffer value, FieldType fieldType, String columnName, Type columnType) {
            this.value = value;
            this.fieldType = fieldType;
            this.columnName = columnName;
            this.columnType = columnType;
            this.size = value.limit() - value.position();
        }

        @Override
        public final boolean isNull() {
            return this.size == 0;
        }

        @Override
        public boolean getBoolean() {
            this.checkEnoughBytes();
            switch (this.fieldType) {
                case BYTE: {
                    return this.value.get() != 0;
                }
                case SHORT: {
                    return this.value.getShort() != 0;
                }
                case INT: {
                    return this.value.getInt() != 0;
                }
                case LONG: {
                    return this.value.getLong() != 0L;
                }
            }
            throw new PrestoException((ErrorCodeSupplier)DecoderErrorCode.DECODER_CONVERSION_NOT_SUPPORTED, String.format("conversion '%s' to boolean not supported", new Object[]{this.fieldType}));
        }

        @Override
        public long getLong() {
            this.checkEnoughBytes();
            switch (this.fieldType) {
                case BYTE: {
                    return this.value.get();
                }
                case SHORT: {
                    return this.value.getShort();
                }
                case INT: {
                    return this.value.getInt();
                }
                case LONG: {
                    return this.value.getLong();
                }
            }
            throw new PrestoException((ErrorCodeSupplier)DecoderErrorCode.DECODER_CONVERSION_NOT_SUPPORTED, String.format("conversion '%s' to long not supported", new Object[]{this.fieldType}));
        }

        @Override
        public double getDouble() {
            this.checkEnoughBytes();
            switch (this.fieldType) {
                case FLOAT: {
                    return this.value.getFloat();
                }
                case DOUBLE: {
                    return this.value.getDouble();
                }
            }
            throw new PrestoException((ErrorCodeSupplier)DecoderErrorCode.DECODER_CONVERSION_NOT_SUPPORTED, String.format("conversion '%s' to double not supported", new Object[]{this.fieldType}));
        }

        private void checkEnoughBytes() {
            Preconditions.checkState((this.size >= this.fieldType.getSize() ? 1 : 0) != 0, (String)"minimum byte size for column '%s' is %s, found %s,", (Object)this.columnName, (Object)this.fieldType.getSize(), (Object)this.size);
        }

        @Override
        public Slice getSlice() {
            Slice slice = Slices.wrappedBuffer((ByteBuffer)this.value.slice());
            return Varchars.truncateToLength((Slice)slice, (Type)this.columnType);
        }
    }

    private static enum FieldType {
        BYTE(8),
        SHORT(16),
        INT(32),
        LONG(64),
        FLOAT(32),
        DOUBLE(64);

        private final int size;

        private FieldType(int bitSize) {
            this.size = bitSize / 8;
        }

        public int getSize() {
            return this.size;
        }
    }
}

