/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.password;

import com.facebook.airlift.http.server.BasicPrincipal;
import com.facebook.airlift.log.Logger;
import com.facebook.presto.password.LdapConfig;
import com.facebook.presto.password.jndi.JndiUtils;
import com.facebook.presto.spi.security.AccessDeniedException;
import com.facebook.presto.spi.security.PasswordAuthenticator;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.VerifyException;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.UncheckedExecutionException;
import java.security.Principal;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import javax.inject.Inject;
import javax.naming.AuthenticationException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.DirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

public class LdapAuthenticator
implements PasswordAuthenticator {
    private static final Logger log = Logger.get(LdapAuthenticator.class);
    private final String userBindSearchPattern;
    private final Optional<String> groupAuthorizationSearchPattern;
    private final Optional<String> userBaseDistinguishedName;
    private final Map<String, String> basicEnvironment;
    private final LoadingCache<Credentials, Principal> authenticationCache;

    @Inject
    public LdapAuthenticator(LdapConfig serverConfig) {
        String ldapUrl = Objects.requireNonNull(serverConfig.getLdapUrl(), "ldapUrl is null");
        this.userBindSearchPattern = Objects.requireNonNull(serverConfig.getUserBindSearchPattern(), "userBindSearchPattern is null");
        this.groupAuthorizationSearchPattern = Optional.ofNullable(serverConfig.getGroupAuthorizationSearchPattern());
        this.userBaseDistinguishedName = Optional.ofNullable(serverConfig.getUserBaseDistinguishedName());
        if (this.groupAuthorizationSearchPattern.isPresent()) {
            Preconditions.checkState((boolean)this.userBaseDistinguishedName.isPresent(), (Object)"Base distinguished name (DN) for user is null");
        }
        ImmutableMap environment = ImmutableMap.builder().put((Object)"java.naming.factory.initial", (Object)"com.sun.jndi.ldap.LdapCtxFactory").put((Object)"java.naming.provider.url", (Object)ldapUrl).build();
        LdapAuthenticator.checkEnvironment((Map<String, String>)environment);
        this.basicEnvironment = environment;
        this.authenticationCache = CacheBuilder.newBuilder().expireAfterWrite(serverConfig.getLdapCacheTtl().toMillis(), TimeUnit.MILLISECONDS).build(CacheLoader.from(this::authenticate));
    }

    public Principal createAuthenticatedPrincipal(String user, String password) {
        try {
            return (Principal)this.authenticationCache.getUnchecked((Object)new Credentials(user, password));
        }
        catch (UncheckedExecutionException e) {
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), AccessDeniedException.class);
            throw e;
        }
    }

    private Principal authenticate(Credentials credentials) {
        return this.authenticate(credentials.getUser(), credentials.getPassword());
    }

    private Principal authenticate(String user, String password) {
        Map<String, String> environment = this.createEnvironment(user, password);
        DirContext context = null;
        try {
            context = JndiUtils.createDirContext(environment);
            this.checkForGroupMembership(user, context);
            log.debug("Authentication successful for user [%s]", new Object[]{user});
            BasicPrincipal basicPrincipal = new BasicPrincipal(user);
            return basicPrincipal;
        }
        catch (AuthenticationException e) {
            log.debug("Authentication failed for user [%s]: %s", new Object[]{user, e.getMessage()});
            throw new AccessDeniedException("Invalid credentials");
        }
        catch (NamingException e) {
            log.debug((Throwable)e, "Authentication error for user [%s]", new Object[]{user});
            throw new RuntimeException("Authentication error");
        }
        finally {
            if (context != null) {
                LdapAuthenticator.closeContext(context);
            }
        }
    }

    private Map<String, String> createEnvironment(String user, String password) {
        return ImmutableMap.builder().putAll(this.basicEnvironment).put((Object)"java.naming.security.authentication", (Object)"simple").put((Object)"java.naming.security.principal", (Object)this.createPrincipal(user)).put((Object)"java.naming.security.credentials", (Object)password).build();
    }

    private String createPrincipal(String user) {
        return LdapAuthenticator.replaceUser(this.userBindSearchPattern, user);
    }

    private void checkForGroupMembership(String user, DirContext context) {
        boolean authorized;
        if (!this.groupAuthorizationSearchPattern.isPresent()) {
            return;
        }
        String userBase = this.userBaseDistinguishedName.orElseThrow(VerifyException::new);
        String searchFilter = LdapAuthenticator.replaceUser(this.groupAuthorizationSearchPattern.get(), user);
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(2);
        try {
            NamingEnumeration<SearchResult> search = context.search(userBase, searchFilter, searchControls);
            authorized = search.hasMoreElements();
            search.close();
        }
        catch (NamingException e) {
            log.debug("Authentication error for user [%s]: %s", new Object[]{user, e.getMessage()});
            throw new RuntimeException("Authentication error");
        }
        if (!authorized) {
            String message = String.format("User [%s] not a member of the authorized group", user);
            log.debug(message);
            throw new AccessDeniedException(message);
        }
    }

    private static String replaceUser(String pattern, String user) {
        return pattern.replaceAll("\\$\\{USER}", user);
    }

    private static void checkEnvironment(Map<String, String> environment) {
        try {
            LdapAuthenticator.closeContext(JndiUtils.createDirContext(environment));
        }
        catch (NamingException e) {
            throw new RuntimeException(e);
        }
    }

    private static void closeContext(DirContext context) {
        try {
            context.close();
        }
        catch (NamingException namingException) {
            // empty catch block
        }
    }

    private static class Credentials {
        private final String user;
        private final String password;

        private Credentials(String user, String password) {
            this.user = Objects.requireNonNull(user);
            this.password = Objects.requireNonNull(password);
        }

        public String getUser() {
            return this.user;
        }

        public String getPassword() {
            return this.password;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Credentials that = (Credentials)o;
            return Objects.equals(this.user, that.user) && Objects.equals(this.password, that.password);
        }

        public int hashCode() {
            return Objects.hash(this.user, this.password);
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("user", (Object)this.user).add("password", (Object)this.password).toString();
        }
    }
}

