/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.spiller;

import com.facebook.airlift.log.Logger;
import com.facebook.presto.common.io.DataOutput;
import com.facebook.presto.common.io.DataSink;
import com.facebook.presto.common.io.OutputStreamDataSink;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.storage.StorageCapabilities;
import com.facebook.presto.spi.storage.TempDataOperationContext;
import com.facebook.presto.spi.storage.TempDataSink;
import com.facebook.presto.spi.storage.TempStorage;
import com.facebook.presto.spi.storage.TempStorageContext;
import com.facebook.presto.spi.storage.TempStorageFactory;
import com.facebook.presto.spi.storage.TempStorageHandle;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import javax.annotation.concurrent.GuardedBy;

public class LocalTempStorage
implements TempStorage {
    public static final String NAME = "local";
    public static final String TEMP_STORAGE_PATH = "temp-storage.path";
    private static final Logger log = Logger.get(LocalTempStorage.class);
    private static final String SPILL_FILE_PREFIX = "spill";
    private static final String SPILL_FILE_SUFFIX = ".bin";
    private static final String SPILL_FILE_GLOB = "spill*.bin";
    private final List<Path> spillPaths;
    private final double maxUsedSpaceThreshold;
    @GuardedBy(value="this")
    private int roundRobinIndex;

    public LocalTempStorage(List<Path> spillPaths, double maxUsedSpaceThreshold) {
        this.spillPaths = ImmutableList.copyOf((Collection)Objects.requireNonNull(spillPaths, "spillPaths is null"));
        this.maxUsedSpaceThreshold = maxUsedSpaceThreshold;
        this.initialize();
    }

    private void initialize() {
        this.spillPaths.forEach(path -> {
            try {
                Files.createDirectories(path, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new IllegalArgumentException(String.format("could not create spill path %s; adjust experimental.spiller-spill-path config property or filesystem permissions", path), e);
            }
            if (!path.toFile().canWrite()) {
                throw new IllegalArgumentException(String.format("spill path %s is not writable; adjust experimental.spiller-spill-path config property or filesystem permissions", path));
            }
        });
        this.spillPaths.forEach(LocalTempStorage::cleanupOldSpillFiles);
    }

    public TempDataSink create(TempDataOperationContext context) throws IOException {
        Path path = Files.createTempFile(this.getNextSpillPath(), SPILL_FILE_PREFIX, SPILL_FILE_SUFFIX, new FileAttribute[0]);
        return new LocalTempDataSink(path);
    }

    public InputStream open(TempDataOperationContext context, TempStorageHandle handle) throws IOException {
        return Files.newInputStream(((LocalTempStorageHandle)handle).getFilePath(), new OpenOption[0]);
    }

    public void remove(TempDataOperationContext context, TempStorageHandle handle) throws IOException {
        Files.delete(((LocalTempStorageHandle)handle).getFilePath());
    }

    public byte[] serializeHandle(TempStorageHandle storageHandle) {
        URI uri = ((LocalTempStorageHandle)storageHandle).getFilePath().toUri();
        return uri.toString().getBytes(StandardCharsets.UTF_8);
    }

    public TempStorageHandle deserialize(byte[] serializedStorageHandle) {
        String uriString = new String(serializedStorageHandle, StandardCharsets.UTF_8);
        try {
            return new LocalTempStorageHandle(Paths.get(new URI(uriString)));
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException("Invalid URI: " + uriString, e);
        }
    }

    public List<StorageCapabilities> getStorageCapabilities() {
        return ImmutableList.of();
    }

    private static void cleanupOldSpillFiles(Path path) {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(path, SPILL_FILE_GLOB);){
            stream.forEach(spillFile -> {
                try {
                    log.info("Deleting old spill file: " + spillFile);
                    Files.delete(spillFile);
                }
                catch (Exception e) {
                    log.warn("Could not cleanup old spill file: " + spillFile);
                }
            });
        }
        catch (IOException e) {
            log.warn((Throwable)e, "Error cleaning spill files");
        }
    }

    private synchronized Path getNextSpillPath() {
        int spillPathsCount = this.spillPaths.size();
        for (int i = 0; i < spillPathsCount; ++i) {
            int pathIndex = (this.roundRobinIndex + i) % spillPathsCount;
            Path path = this.spillPaths.get(pathIndex);
            if (!this.hasEnoughDiskSpace(path)) continue;
            this.roundRobinIndex = (this.roundRobinIndex + i + 1) % spillPathsCount;
            return path;
        }
        if (this.spillPaths.isEmpty()) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.OUT_OF_SPILL_SPACE, "No spill paths configured");
        }
        throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.OUT_OF_SPILL_SPACE, "No free space available for spill");
    }

    private boolean hasEnoughDiskSpace(Path path) {
        try {
            FileStore fileStore = Files.getFileStore(path);
            return (double)fileStore.getUsableSpace() > (double)fileStore.getTotalSpace() * (1.0 - this.maxUsedSpaceThreshold);
        }
        catch (IOException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.OUT_OF_SPILL_SPACE, "Cannot determine free space for spill", (Throwable)e);
        }
    }

    private static class LocalTempDataSink
    implements TempDataSink {
        private final DataSink sink;
        private final Path path;

        public LocalTempDataSink(Path path) throws IOException {
            this.path = Objects.requireNonNull(path, "path is null");
            this.sink = new OutputStreamDataSink(Files.newOutputStream(path, StandardOpenOption.APPEND));
        }

        public TempStorageHandle commit() throws IOException {
            this.sink.close();
            return new LocalTempStorageHandle(this.path);
        }

        public void rollback() throws IOException {
            this.commit();
            Files.delete(this.path);
        }

        public long size() {
            return this.sink.size();
        }

        public long getRetainedSizeInBytes() {
            return this.sink.getRetainedSizeInBytes();
        }

        public void write(List<DataOutput> outputData) throws IOException {
            this.sink.write(outputData);
        }

        public void close() throws IOException {
            this.sink.close();
        }
    }

    private static class LocalTempStorageHandle
    implements TempStorageHandle {
        private final Path filePath;

        public LocalTempStorageHandle(Path filePath) {
            this.filePath = Objects.requireNonNull(filePath, "filePath is null");
        }

        public Path getFilePath() {
            return this.filePath;
        }

        public String toString() {
            return this.filePath.toString();
        }
    }

    public static class Factory
    implements TempStorageFactory {
        public String getName() {
            return LocalTempStorage.NAME;
        }

        public TempStorage create(Map<String, String> config, TempStorageContext context) {
            String configPaths = config.get(LocalTempStorage.TEMP_STORAGE_PATH);
            Preconditions.checkState((configPaths != null ? 1 : 0) != 0, (String)"Local temp storage configuration must contain the '%s' property", (Object)LocalTempStorage.TEMP_STORAGE_PATH);
            ImmutableList pathsSplit = ImmutableList.copyOf((Iterable)Splitter.on((String)",").trimResults().omitEmptyStrings().split((CharSequence)config.get(LocalTempStorage.TEMP_STORAGE_PATH)));
            List tempStoragePaths = (List)pathsSplit.stream().map(x$0 -> Paths.get(x$0, new String[0])).collect(ImmutableList.toImmutableList());
            return new LocalTempStorage(tempStoragePaths, 1.0);
        }
    }
}

