/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.geospatial;

import com.esri.core.geometry.Envelope;
import com.esri.core.geometry.Geometry;
import com.esri.core.geometry.Operator;
import com.esri.core.geometry.OperatorIntersects;
import com.esri.core.geometry.Point;
import com.esri.core.geometry.ogc.OGCGeometry;
import com.facebook.presto.geospatial.BingTile;
import com.facebook.presto.geospatial.GeometryUtils;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.function.SqlType;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.Slice;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.PriorityQueue;
import java.util.function.Consumer;

public class BingTileUtils {
    static final int TILE_PIXELS = 256;
    static final double EARTH_RADIUS_KM = 6371.01;
    static final double MAX_LATITUDE = 85.05112878;
    static final double MIN_LATITUDE = -85.05112878;
    static final double MIN_LONGITUDE = -180.0;
    static final double MAX_LONGITUDE = 180.0;
    static final String LATITUDE_SPAN_OUT_OF_RANGE = String.format("Latitude span for the geometry must be in [%.2f, %.2f] range", -85.05112878, 85.05112878);
    static final String LATITUDE_OUT_OF_RANGE = "Latitude must be between -85.05112878 and 85.05112878";
    static final String LONGITUDE_SPAN_OUT_OF_RANGE = String.format("Longitude span for the geometry must be in [%.2f, %.2f] range", -180.0, 180.0);
    static final String LONGITUDE_OUT_OF_RANGE = "Longitude must be between -180.0 and 180.0";
    private static final String QUAD_KEY_TOO_LONG = "QuadKey must be 23 characters or less";
    private static final String ZOOM_LEVEL_TOO_SMALL = "Zoom level must be >= 0";
    private static final String ZOOM_LEVEL_TOO_LARGE = "Zoom level must be <= 23";
    private static final int MAX_COVERING_COUNT = 1000000;

    private BingTileUtils() {
    }

    static void checkZoomLevel(long zoomLevel) {
        BingTileUtils.checkCondition(zoomLevel >= 0L, ZOOM_LEVEL_TOO_SMALL, new Object[0]);
        BingTileUtils.checkCondition(zoomLevel <= 23L, ZOOM_LEVEL_TOO_LARGE, new Object[0]);
    }

    static void checkCoordinate(long coordinate, long zoomLevel) {
        BingTileUtils.checkCondition(coordinate >= 0L && coordinate < (long)(1 << (int)zoomLevel), "XY coordinates for a Bing tile at zoom level %s must be within [0, %s) range", zoomLevel, 1 << (int)zoomLevel);
    }

    static void checkQuadKey(@SqlType(value="varchar") Slice quadkey) {
        BingTileUtils.checkCondition(quadkey.length() <= 23, QUAD_KEY_TOO_LONG, new Object[0]);
    }

    static void checkLatitude(double latitude, String errorMessage) {
        BingTileUtils.checkCondition(latitude >= -85.05112878 && latitude <= 85.05112878, errorMessage, new Object[0]);
    }

    static void checkLongitude(double longitude, String errorMessage) {
        BingTileUtils.checkCondition(longitude >= -180.0 && longitude <= 180.0, errorMessage, new Object[0]);
    }

    static void checkCondition(boolean condition, String formatString, Object ... args) {
        if (!condition) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format(formatString, args));
        }
    }

    public static double tileXToLongitude(int tileX, int zoomLevel) {
        int mapTileSize = 1 << zoomLevel;
        double x = BingTileUtils.clip(tileX, 0.0, mapTileSize) / (double)mapTileSize - 0.5;
        return 360.0 * x;
    }

    public static double tileYToLatitude(int tileY, int zoomLevel) {
        int mapTileSize = 1 << zoomLevel;
        double y = 0.5 - BingTileUtils.clip(tileY, 0.0, mapTileSize) / (double)mapTileSize;
        return 90.0 - 360.0 * Math.atan(Math.exp(-y * 2.0 * Math.PI)) / Math.PI;
    }

    static Point tileXYToLatitudeLongitude(int tileX, int tileY, int zoomLevel) {
        return new Point(BingTileUtils.tileXToLongitude(tileX, zoomLevel), BingTileUtils.tileYToLatitude(tileY, zoomLevel));
    }

    static Envelope tileToEnvelope(BingTile tile) {
        double minX = BingTileUtils.tileXToLongitude(tile.getX(), tile.getZoomLevel());
        double maxX = BingTileUtils.tileXToLongitude(tile.getX() + 1, tile.getZoomLevel());
        double minY = BingTileUtils.tileYToLatitude(tile.getY(), tile.getZoomLevel());
        double maxY = BingTileUtils.tileYToLatitude(tile.getY() + 1, tile.getZoomLevel());
        return new Envelope(minX, minY, maxX, maxY);
    }

    static double clip(double n, double minValue, double maxValue) {
        return Math.min(Math.max(n, minValue), maxValue);
    }

    static long mapSize(int zoomLevel) {
        return 256L << zoomLevel;
    }

    static BingTile latitudeLongitudeToTile(double latitude, double longitude, int zoomLevel) {
        long mapSize = BingTileUtils.mapSize(zoomLevel);
        int tileX = BingTileUtils.longitudeToTileX(longitude, mapSize);
        int tileY = BingTileUtils.latitudeToTileY(latitude, mapSize);
        return BingTile.fromCoordinates(tileX, tileY, zoomLevel);
    }

    static int longitudeToTileX(double longitude, long mapSize) {
        double x = (longitude + 180.0) / 360.0;
        return BingTileUtils.axisToCoordinates(x, mapSize);
    }

    static int latitudeToTileY(double latitude, long mapSize) {
        double sinLatitude = Math.sin(latitude * Math.PI / 180.0);
        double y = 0.5 - Math.log((1.0 + sinLatitude) / (1.0 - sinLatitude)) / (Math.PI * 4);
        return BingTileUtils.axisToCoordinates(y, mapSize);
    }

    private static int axisToCoordinates(double axis, long mapSize) {
        int tileAxis = (int)BingTileUtils.clip(axis * (double)mapSize, 0.0, mapSize - 1L);
        return tileAxis / 256;
    }

    private static List<BingTile> findRawTileCovering(OGCGeometry ogcGeometry, int maxZoom) {
        Envelope envelope = GeometryUtils.getEnvelope((OGCGeometry)ogcGeometry);
        Optional<List<BingTile>> trivialResult = BingTileUtils.handleTrivialCases(envelope, maxZoom);
        if (trivialResult.isPresent()) {
            return trivialResult.get();
        }
        GeometryUtils.accelerateGeometry((OGCGeometry)ogcGeometry, (Operator)OperatorIntersects.local(), (Geometry.GeometryAccelerationDegree)Geometry.GeometryAccelerationDegree.enumMedium);
        ArrayDeque stack = new ArrayDeque();
        Consumer<BingTile> addIntersecting = tile -> {
            TilingEntry tilingEntry = new TilingEntry((BingTile)tile);
            if (BingTileUtils.satisfiesTileEdgeCondition(envelope, tilingEntry) && ogcGeometry.intersects(tilingEntry.polygon)) {
                stack.push(tilingEntry);
            }
        };
        ImmutableList.of((Object)BingTile.fromCoordinates(0, 0, 1), (Object)BingTile.fromCoordinates(0, 1, 1), (Object)BingTile.fromCoordinates(1, 0, 1), (Object)BingTile.fromCoordinates(1, 1, 1)).forEach(addIntersecting);
        ArrayList<BingTile> outputTiles = new ArrayList<BingTile>();
        while (!stack.isEmpty()) {
            TilingEntry entry = (TilingEntry)stack.pop();
            if (entry.tile.getZoomLevel() == maxZoom || ogcGeometry.contains(entry.polygon)) {
                outputTiles.add(entry.tile);
                continue;
            }
            entry.tile.findChildren().forEach(addIntersecting);
            BingTileUtils.checkCondition(outputTiles.size() + stack.size() <= 1000000, "The zoom level is too high or the geometry is too large to compute a set of covering Bing tiles. Please use a lower zoom level, or tile only a section of the geometry.", new Object[0]);
        }
        return outputTiles;
    }

    private static Optional<List<BingTile>> handleTrivialCases(Envelope envelope, int zoom) {
        BingTileUtils.checkZoomLevel(zoom);
        if (envelope.isEmpty()) {
            return Optional.of(ImmutableList.of());
        }
        BingTileUtils.checkLatitude(envelope.getYMin(), LATITUDE_SPAN_OUT_OF_RANGE);
        BingTileUtils.checkLatitude(envelope.getYMax(), LATITUDE_SPAN_OUT_OF_RANGE);
        BingTileUtils.checkLongitude(envelope.getXMin(), LONGITUDE_SPAN_OUT_OF_RANGE);
        BingTileUtils.checkLongitude(envelope.getXMax(), LONGITUDE_SPAN_OUT_OF_RANGE);
        if (zoom == 0) {
            return Optional.of(ImmutableList.of((Object)BingTile.fromCoordinates(0, 0, 0)));
        }
        if (envelope.getXMax() == envelope.getXMin() && envelope.getYMax() == envelope.getYMin()) {
            return Optional.of(ImmutableList.of((Object)BingTileUtils.latitudeLongitudeToTile(envelope.getYMax(), envelope.getXMax(), zoom)));
        }
        return Optional.empty();
    }

    private static boolean satisfiesTileEdgeCondition(Envelope query, TilingEntry entry) {
        BingTile tile = entry.tile;
        int maxXY = (1 << tile.getZoomLevel()) - 1;
        if (tile.getY() < maxXY && query.getYMax() == entry.envelope.getYMin()) {
            return false;
        }
        return tile.getX() >= maxXY || query.getXMin() != entry.envelope.getXMax();
    }

    public static List<BingTile> findDissolvedTileCovering(OGCGeometry ogcGeometry, int maxZoom) {
        List<BingTile> rawTiles = BingTileUtils.findRawTileCovering(ogcGeometry, maxZoom);
        if (rawTiles.isEmpty()) {
            return rawTiles;
        }
        ArrayList<BingTile> results = new ArrayList<BingTile>(rawTiles.size());
        HashSet<BingTile> candidates = new HashSet<BingTile>(4);
        Comparator<BingTile> tileComparator = Comparator.comparing(BingTile::getZoomLevel).thenComparing(BingTile::toQuadKey).reversed();
        PriorityQueue<BingTile> queue = new PriorityQueue<BingTile>(rawTiles.size(), tileComparator);
        queue.addAll(rawTiles);
        while (!queue.isEmpty()) {
            BingTile candidate = queue.poll();
            if (candidate.getZoomLevel() == 0) {
                results.add(candidate);
                continue;
            }
            BingTile parent = candidate.findParent();
            candidates.add(candidate);
            while (!queue.isEmpty() && queue.peek().findParent().equals(parent)) {
                candidates.add(queue.poll());
            }
            if (candidates.size() == 4) {
                queue.add(parent);
            } else {
                results.addAll(candidates);
            }
            candidates.clear();
        }
        return results;
    }

    public static List<BingTile> findMinimalTileCovering(OGCGeometry ogcGeometry, int zoom) {
        ArrayList<BingTile> outputTiles = new ArrayList<BingTile>();
        ArrayDeque<BingTile> stack = new ArrayDeque<BingTile>(BingTileUtils.findRawTileCovering(ogcGeometry, zoom));
        while (!stack.isEmpty()) {
            BingTile thisTile = (BingTile)stack.pop();
            outputTiles.addAll(thisTile.findChildren(zoom));
            BingTileUtils.checkCondition(outputTiles.size() + stack.size() <= 1000000, "The zoom level is too high or the geometry is too large to compute a set of covering Bing tiles. Please use a lower zoom level, or tile only a section of the geometry.", new Object[0]);
        }
        return outputTiles;
    }

    public static List<BingTile> findMinimalTileCovering(Envelope envelope, int zoom) {
        Optional<List<BingTile>> maybeResult = BingTileUtils.handleTrivialCases(envelope, zoom);
        if (maybeResult.isPresent()) {
            return maybeResult.get();
        }
        BingTile seTile = BingTileUtils.latitudeLongitudeToTile(envelope.getYMin(), envelope.getXMax(), zoom);
        BingTile nwTile = BingTileUtils.latitudeLongitudeToTile(envelope.getYMax(), envelope.getXMin(), zoom);
        int minY = nwTile.getY();
        int minX = nwTile.getX();
        int maxY = seTile.getY();
        int maxX = seTile.getX();
        int numTiles = (maxX - minX + 1) * (maxY - minY + 1);
        BingTileUtils.checkCondition(numTiles <= 1000000, "The zoom level is too high or the geometry is too large to compute a set of covering Bing tiles. Please use a lower zoom level, or tile only a section of the geometry.", new Object[0]);
        ArrayList<BingTile> results = new ArrayList<BingTile>(numTiles);
        for (int y = minY; y <= maxY; ++y) {
            for (int x = minX; x <= maxX; ++x) {
                results.add(BingTile.fromCoordinates(x, y, zoom));
            }
        }
        return results;
    }

    private static class TilingEntry {
        private final BingTile tile;
        private final Envelope envelope;
        private final OGCGeometry polygon;

        public TilingEntry(BingTile tile) {
            this.tile = tile;
            this.envelope = BingTileUtils.tileToEnvelope(tile);
            this.polygon = OGCGeometry.createFromEsriGeometry((Geometry)this.envelope, null);
        }
    }
}

